package com.vaadin.copilot.javarewriter;

import static com.vaadin.copilot.javarewriter.DataEntityRecordRewriter.DEFAULT_ENTITY_RECORD_NAME;

import java.lang.reflect.Constructor;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.copilot.javarewriter.custom.CustomComponentHandle;
import com.vaadin.copilot.javarewriter.custom.CustomComponentHandler;

import com.github.javaparser.ParserConfiguration;
import com.github.javaparser.Range;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.comments.BlockComment;
import com.github.javaparser.ast.comments.Comment;
import com.github.javaparser.ast.comments.LineComment;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.BooleanLiteralExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithArguments;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.Statement;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import com.github.javaparser.resolution.declarations.ResolvedReferenceTypeDeclaration;
import com.github.javaparser.resolution.model.SymbolReference;
import com.github.javaparser.symbolsolver.JavaSymbolSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.ClassLoaderTypeSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.CombinedTypeSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.ReflectionTypeSolver;
import org.apache.commons.lang3.StringUtils;

/**
 * Rewrites Java source code to add or replace constructor parameters, method
 * invocations and more.
 */
public class JavaRewriter {

    private static final String UNKNOWN_WHERE = "Unknown where: ";
    private static final String CHART_SERIES_CLASS = "ChartSeries";
    private static final String ADD_CLASS_NAMES_METHOD = "addClassNames";
    private static final String SET_SPACING_METHOD = "setSpacing";
    private static final String ADD_ITEM_METHOD = "addItem";
    private final String source;
    private final JavaRewriterObserver observer = new JavaRewriterObserver();

    /**
     * A code snippet to be inserted into the source code.
     *
     * @param code
     *            the code snippet
     */
    public record Code(String code) {
    }

    /**
     * Holder for a setter name and associated value
     */
    public record SetterAndValue(String setter, Object value) {
    }

    /**
     * Information about extracting an inline variable to local variable
     */
    public record ExtractInlineVariableResult(BlockStmt blockStmt, String newVariableName, int index) {
    }

    /**
     * Information about a renamed variable.
     *
     * @param variableRenamedTo
     *            the new name for the variable
     */
    public record ReplaceResult(String variableRenamedTo) {

    }

    /**
     * The result of a duplicate operation
     *
     * @param nameMapping
     *            a map from old component name to new component name
     * @param childAddCalls
     *            a list of add calls from the parent to ddd children
     * @param variableDeclaration
     *            the variable declaration of the new component when it is declared
     *            as local variable to manage children attachment in constructor
     * @param assignExpr
     *            the assign expression of the new component when it is declared as
     *            field to manage children attachment in constructor
     */
    public record DuplicateInfo(Map<String, String> nameMapping, List<MethodCallExpr> childAddCalls,
            VariableDeclarationExpr variableDeclaration, AssignExpr assignExpr) {
    }

    public record AddTemplateOptions(boolean javaFieldsForLeafComponents) {

    }

    /**
     * Where to add a component
     */
    public enum Where {
        BEFORE, APPEND
    }

    public enum AlignmentMode {
        ALIGN_ITEMS, SELF_HORIZONTALLY, SELF_VERTICALLY
    }

    private final ParserConfiguration parserConfiguration = new ParserConfiguration();
    protected CompilationUnit compilationUnit;

    /**
     * Creates a new JavaRewriter instance.
     *
     * @param source
     *            the Java source code to rewrite
     */
    public JavaRewriter(String source) {
        this(source, false);
    }

    /**
     * Creates a new JavaRewriter instance.
     *
     * @param source
     *            the Java source code to rewrite
     * @param solveJdkClassesBySimpleName
     *            true to try to resolve unqualified types using a set of standard
     *            JDK packages∫
     */
    public JavaRewriter(String source, boolean solveJdkClassesBySimpleName) {
        this.source = source;
        parseSource(source, solveJdkClassesBySimpleName);
    }

    private void parseSource(String source, boolean solveJdkClassesBySimpleName) {
        parserConfiguration.setLanguageLevel(ParserConfiguration.LanguageLevel.JAVA_17);
        CombinedTypeSolver combinedTypeSolver = new CombinedTypeSolver();
        combinedTypeSolver.add(new ReflectionTypeSolver(false));
        if (solveJdkClassesBySimpleName) {
            combinedTypeSolver.add(new ClassLoaderTypeSolver(ClassLoaderTypeSolver.class.getClassLoader()) {
                @Override
                public SymbolReference<ResolvedReferenceTypeDeclaration> tryToSolveType(String name) {
                    if (name.contains(".")) {
                        return SymbolReference.unsolved();
                    }

                    // java.lang is already handled by the default solver
                    for (String pkg : Arrays.asList(LocalDateTime.class.getPackageName(), Map.class.getPackageName())) {
                        SymbolReference<ResolvedReferenceTypeDeclaration> ref = super.tryToSolveType(pkg + "." + name);
                        if (ref.isSolved()) {
                            return ref;
                        }
                    }
                    return SymbolReference.unsolved();
                }
            });
        }
        JavaSymbolSolver symbolSolver = new JavaSymbolSolver(combinedTypeSolver);
        parserConfiguration.setSymbolResolver(symbolSolver);

        StaticJavaParser.setConfiguration(parserConfiguration);
        this.compilationUnit = StaticJavaParser.parse(source);
        this.compilationUnit.registerForSubtree(this.observer);
    }

    public CompilationUnit getCompilationUnit() {
        return compilationUnit;
    }

    public String getResult() {
        return JavaRewriterMerger.apply(observer.getAddedOrModifiedNodes(), observer.getRemovedRanges(), source);
    }

    public int getFirstModifiedRow() {
        int row = 1;
        for (int i = 0; i < source.length(); i++) {
            if (source.charAt(i) != getResult().charAt(i)) {
                return row;
            }
            if (source.charAt(i) == '\n') {
                row++;
            }
        }
        return -1;
    }

    /**
     * Replaces a constructor parameter (if it is mapped to the given setter
     * function) or a function call in the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function to replace or add, if the constructor
     *            parameter is not found
     * @param value
     *            the new value for the constructor parameter or function call
     * @return the new variable name if the variable name was auto-generated and
     *         changed, otherwise null
     */
    public ReplaceResult replaceFunctionCall(ComponentInfo componentInfo, String function, Object value) {
        String currentText = null;
        String currentLabel = null;

        boolean affectsVariableName = JavaRewriterUtil.functionAffectsVariableName(function);
        boolean variableNameAutoGenerated = false;
        if (affectsVariableName) {
            Object currentTextProperty = getPropertyValue(componentInfo, "text");
            Object currentLabelProperty = getPropertyValue(componentInfo, "label");
            currentText = currentTextProperty instanceof String text ? text : null;
            currentLabel = currentLabelProperty instanceof String label ? label : null;
            variableNameAutoGenerated = JavaRewriterUtil.isVariableNameAutoGenerated(componentInfo, currentText,
                    currentLabel);
        }

        boolean replaced = replaceConstructorParam(componentInfo, function, value);
        if (!replaced) {
            replaceOrAddCall(componentInfo, function, value);
        }

        if (variableNameAutoGenerated) {
            String newText = function.equals("setText") && value instanceof String string ? string : null;
            String newLabel = function.equals("setLabel") && value instanceof String string ? string : null;
            // If we replace a static string with a translations call, we don't want to
            // rename the variable
            if (newText != null || newLabel != null) {
                return new ReplaceResult(JavaRewriterUtil.regenerateVariableName(componentInfo, newText, newLabel));
            }
        }
        return new ReplaceResult(null);
    }

    /**
     * Adds the given import if it is not already imported.
     *
     * @param qualifiedName
     *            the qualified name of the import
     * @param isStatic
     *            whether the import is static
     * @param isAsterisk
     *            whether the import end with ".*"
     */
    public void addImport(String qualifiedName, boolean isStatic, boolean isAsterisk) {
        JavaRewriterUtil.addImport(compilationUnit, qualifiedName, isStatic, isAsterisk);
    }

    /**
     * Replaces a constructor parameter (if it is mapped to the given setter
     * function) in the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the setter function
     * @param value
     *            the new value for the constructor parameter
     * @return {@code true} if the replacement was successful, {@code false}
     *         otherwise
     */
    private boolean replaceConstructorParam(ComponentInfo componentInfo, String function, Object value) {
        ObjectCreationExpr objectCreationExpr = componentInfo.objectCreationExpr();

        // Find constructor based on number of arguments
        Optional<Constructor<?>> constructor = JavaRewriterUtil.findConstructor(componentInfo.type(),
                objectCreationExpr);
        Optional<Map<Integer, String>> mappings = constructor.map(c -> ConstructorAnalyzer.get().getMappings(c));
        if (mappings.isPresent()) {
            Optional<Map.Entry<Integer, String>> mapping = mappings.get().entrySet().stream()
                    .filter(entry -> entry.getValue().equals(function)).findFirst();
            if (mapping.isPresent()) {
                objectCreationExpr.setArgument(mapping.get().getKey(), JavaRewriterUtil.toExpression(value));
                return true;
            }
        }
        return false;
    }

    /**
     * Adds a function call to the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function to add
     * @param parameters
     *            parameters for the function
     */
    public void addCall(ComponentInfo componentInfo, String function, Object... parameters) {
        doReplaceOrAddCall(componentInfo, function, false, parameters);
    }

    /**
     * Replaces a function call in the source code, if found, otherwise adds the
     * function call.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function call to add or replace
     * @param parameters
     *            new parameters for the function
     */
    public void replaceOrAddCall(ComponentInfo componentInfo, String function, Object... parameters) {
        doReplaceOrAddCall(componentInfo, function, true, parameters);
    }

    private void doReplaceOrAddCall(ComponentInfo componentInfo, String function, boolean replace,
            Object... parameters) {
        List<Expression> parameterExpressions = new ArrayList<>();
        for (Object parameter : parameters) {
            parameterExpressions.add(JavaRewriterUtil.toExpression(parameter));
        }

        List<MethodCallExpr> functionCalls = JavaRewriterUtil.findMethodCallStatements(componentInfo);

        MethodCallExpr functionCall = functionCalls.stream().filter(m -> m.getNameAsString().equals(function))
                .findFirst().orElse(null);
        if (replace && functionCall != null) {
            if (parameterExpressions.size() == 1) {
                functionCall.setArgument(0, parameterExpressions.get(0));
            }

            return;
        }

        JavaRewriterUtil.addFunctionCall(componentInfo, function, parameterExpressions);
    }

    /**
     * Gets the (active) value of a property of a component.
     *
     * <p>
     * The property value is determined by looking for a setter method call in the
     * source code. If the property is not set using a setter, the constructor is
     * checked.
     *
     * <p>
     * If the property is not set using a setter or in the constructor, {@code null}
     * is returned.
     *
     * <p>
     * If the property is set using a method call, the method call expression is
     * returned.
     *
     * @param componentInfo
     *            the component to get the property value from
     * @param property
     *            the property name
     * @return the property value, or null if the property is not set
     */
    public Object getPropertyValue(ComponentInfo componentInfo, String property) {
        String setterName = JavaRewriterUtil.getSetterName(property, componentInfo.type(), false);
        List<MethodCallExpr> functionCalls = JavaRewriterUtil.findMethodCallStatements(componentInfo);
        List<MethodCallExpr> candidates = functionCalls.stream().filter(m -> m.getNameAsString().equals(setterName))
                .toList();
        if (!candidates.isEmpty()) {
            // If there would be multiple calls, the last one is the effective
            // one
            MethodCallExpr setterCall = candidates.get(candidates.size() - 1);
            Expression arg = setterCall.getArguments().get(0);
            return JavaRewriterUtil.fromExpression(arg, setterCall.resolve().getParam(0).getType());
        }

        // Try constructor
        ObjectCreationExpr createExpression = componentInfo.objectCreationExpr();
        if (createExpression != null) {
            Optional<Constructor<?>> maybeConstructor = JavaRewriterUtil.findConstructor(componentInfo.type(),
                    createExpression);
            if (maybeConstructor.isPresent()) {
                Constructor<?> constructor = maybeConstructor.get();
                for (int i = 0; i < constructor.getParameterCount(); i++) {
                    String mappedProperty = JavaRewriterUtil.getMappedProperty(constructor, i);
                    if (setterName.equals(mappedProperty)) {
                        return JavaRewriterUtil.fromExpression(createExpression.getArgument(i),
                                createExpression.resolve().getParam(i).getType());
                    }
                }
            }
        }
        return null;
    }

    /**
     * Gets the (active) styles of a component.
     *
     * @param componentInfo
     *            the component to get the styles of
     * @return the styles, as a list of style names and values
     */
    public List<JavaStyleRewriter.StyleInfo> getStyles(ComponentInfo componentInfo) {
        return JavaStyleRewriter.getStyles(componentInfo);
    }

    /**
     * Sets the given inline style on the given component, replacing an existing
     * style property if present.
     *
     * @param component
     *            the component to set the style on
     * @param property
     *            the style property to set
     * @param value
     *            the style value to set or null to remove the style
     */
    public void setStyle(ComponentInfo component, String property, String value) {
        JavaStyleRewriter.setStyle(component, property, value);
    }

    /**
     * Sets sizing properties of the given component using the Style API. Replaces
     * existing ones and removes the keys which have null values. <br/>
     * For removal, it also looks for Component API to remove such
     * <code>button.setWidth("")</code> calls.
     *
     * @param componentInfo
     *            the component to set style
     * @param changes
     *            changes applied. Having null value for a key means removal,
     *            otherwise update/add applies.
     */
    public void setSizing(ComponentInfo componentInfo, Map<String, String> changes) {
        JavaStyleRewriter.setSizing(componentInfo, changes);
    }

    /**
     * Finds a component in the source code.
     *
     * <p>
     * Note that this will create a constructor if the component is a route class
     * and there are no constructors.
     *
     * @param typeAndSourceLocation
     *            the type and source location of the component
     * @return the component info
     */
    public ComponentInfo findComponentInfo(ComponentTypeAndSourceLocation typeAndSourceLocation) {
        return ComponentInfo.find(typeAndSourceLocation, this);
    }

    /**
     * Deletes a component from the source code.
     *
     * @param componentInfo
     *            the component to delete
     * @return {@code true} if the deletion was successful, {@code false} otherwise
     */
    public boolean delete(ComponentInfo componentInfo) {
        List<MethodCallExpr> functionCalls = JavaRewriterUtil.findMethodCallStatements(componentInfo);
        List<MethodCallExpr> otherMethodCalls = JavaRewriterUtil.findMethodCallNonStatements(componentInfo);

        List<Expression> parameterUsages = JavaRewriterUtil.findParameterUsage(componentInfo);

        Optional<Range> attachRange = componentInfo.attachCall().getNode().getRange();
        if (attachRange.isPresent()) {
            parameterUsages = parameterUsages.stream().filter(parameter -> parameter.getRange().isPresent()
                    && !parameter.getRange().get().overlapsWith(attachRange.get())).toList();
        }

        parameterUsages.forEach(expr -> {
            if (expr.getParentNode().isPresent() && ((Expression) expr.getParentNode().get()).isMethodCallExpr()) {
                MethodCallExpr methodCallExpr = (MethodCallExpr) expr.getParentNode().get();
                Optional<String> classNameOpt = JavaRewriterUtil
                        .findAncestor(methodCallExpr, ClassOrInterfaceDeclaration.class).getFullyQualifiedName();
                if (classNameOpt.isEmpty()) {
                    throw new IllegalArgumentException("Could not find source class");
                }
                int argumentPosition = methodCallExpr.getArgumentPosition(expr);
                boolean arrayArgument = JavaRewriterUtil.isArrayArgument(classNameOpt.get(),
                        methodCallExpr.getNameAsString(), argumentPosition);
                if (arrayArgument) {
                    JavaRewriterUtil.removeArgumentCalls(methodCallExpr, List.of(expr), true);
                } else if (methodCallExpr.getArguments().size() == 1) {
                    JavaRewriterUtil.removeStatement(methodCallExpr);
                } else {
                    throw new IllegalArgumentException("Cannot handle " + methodCallExpr.getNameAsString() + " method");
                }
            }
        });

        functionCalls.forEach(expr -> JavaRewriterUtil.findAncestorOrThrow(expr, Statement.class).remove());
        if (componentInfo.fieldDeclaration() != null) {
            componentInfo.fieldDeclaration().remove();
            if (componentInfo.fieldDeclarationAndAssignment() == null) {
                // Instance created elsewhere, i.e.
                // TextField foo;
                // ...
                // foo = new TextField();
                JavaRewriterUtil.removeStatement(componentInfo.objectCreationExpr());
            }
        } else if (componentInfo.localVariableDeclarator() != null) {
            JavaRewriterUtil.removeStatement(componentInfo.localVariableDeclarator());
        }

        otherMethodCalls.forEach(methodCall -> {
            if (!JavaRewriterUtil.removeFromStringConcatenation(methodCall)) {
                JavaRewriterUtil.removeStatement(methodCall);
            }
        });

        removeAttachCall(componentInfo);

        return true;
    }

    private Optional<Expression> removeAttachCall(ComponentInfo componentInfo) {
        Optional<Expression> addArgument = JavaRewriterUtil.getAttachArgument(componentInfo);
        addArgument.ifPresent(Expression::remove);
        if (componentInfo.attachCall() != null
                && componentInfo.attachCall().getNodeWithArguments().getArguments().isEmpty()) {
            JavaRewriterUtil.removeStatement(componentInfo.attachCall().getNode());
        }
        return addArgument;
    }

    /**
     * Moves a component in the source code.
     *
     * @param component
     *            the component to move
     * @param container
     *            the new container for the component, if where is Where.APPEND.
     * @param reference
     *            the reference component to move the component before, if where is
     *            Where.BEFORE.
     * @param where
     *            where to move the component
     */
    public void moveComponent(ComponentInfo component, ComponentInfo container, ComponentInfo reference, Where where) {
        if (container == null) {
            throw new IllegalArgumentException("Container component must be non-null");
        }
        if (component.equals(container) || component.equals(reference) || container.equals(reference)) {
            throw new IllegalArgumentException("Component, container and reference must be different");
        }
        removeAttachCall(component);

        List<MethodCallExpr> containerFunctionCalls = JavaRewriterUtil.findMethodCallStatements(container);
        String componentFieldOrVariableName = JavaRewriterUtil.getFieldOrVariableName(component);
        // For inline we add the full expression, otherwise the name reference
        var toAdd = componentFieldOrVariableName == null ? component.objectCreationExpr()
                : new NameExpr(componentFieldOrVariableName);

        if (where == Where.APPEND) {
            if (reference != null) {
                throw new IllegalArgumentException("Reference component must be null when appending");
            }

            // Find the last add statement for the container and then add the
            // component after that
            MethodCallExpr added = JavaRewriterUtil.addAfterLastFunctionCall(containerFunctionCalls, "add", toAdd);
            if (added == null) {
                // Container with no function calls, e.g. just
                // var layout = new HorizontalLayout()
                // add after constructor
                InsertionPoint afterCreateExpr = JavaRewriterUtil.findLocationAfter(container.objectCreationExpr());
                var containerVariableName = container.localVariableName() == null ? container.fieldName()
                        : container.localVariableName();
                if (containerVariableName == null) {
                    // Inline defined container
                    ExtractInlineVariableResult extracted = JavaRewriterUtil
                            .extractInlineVariableToLocalVariable(container);
                    if (extracted == null) {
                        throw new IllegalArgumentException("Could not extract inline variable to local variable");
                    }
                    containerVariableName = extracted.newVariableName();
                }
                afterCreateExpr.add(new ExpressionStmt(
                        JavaRewriterUtil.createMethodCall(new NameExpr(containerVariableName), "add", toAdd)));

            }
        } else if (where == Where.BEFORE) {
            if (reference == null) {
                throw new IllegalArgumentException("Reference component must be non-null when moving before");
            }

            Optional<Expression> referenceAddArgument = JavaRewriterUtil
                    .findReference(reference.attachCall().getNodeWithArguments().getArguments(), reference);

            if (referenceAddArgument.isPresent()) {
                int refAddIndex = reference.attachCall().getNodeWithArguments().getArguments()
                        .indexOf(referenceAddArgument.get());
                reference.attachCall().getNodeWithArguments().getArguments().add(refAddIndex, toAdd);

                // We can now end up with having "add" before the component
                // constructor and setters, e.g.

                // Input input1 = new Input();
                // add(input3, input1);
                // Input input2 = new Input();
                // add(input2);
                // Input input3 = new Input();

                // We move the add call down after all functions
                // BUT there can also be add calls between the current
                // add location and where it should end up.
                // If so, they need to be moved also
                // We then get

                // Input input1 = new Input();
                // Input input2 = new Input();
                // Input input3 = new Input();
                // add(input3, input1);
                // add(input2);

                List<MethodCallExpr> componentFunctionCalls = JavaRewriterUtil.findMethodCallStatements(component);
                if (!componentFunctionCalls.isEmpty()) {
                    MethodCallExpr lastCall = componentFunctionCalls.get(componentFunctionCalls.size() - 1);
                    BlockStmt insertBlock = JavaRewriterUtil.findAncestorOrThrow(lastCall, BlockStmt.class);
                    int finalAddLocation = JavaRewriterUtil.findBlockStatementIndex(lastCall) + 1;

                    int attachCallIndex = JavaRewriterUtil.findBlockStatementIndex(reference.attachCall().getNode());

                    List<MethodCallExpr> allAddCalls = containerFunctionCalls.stream()
                            .filter(m -> m.getNameAsString()
                                    .equals(reference.attachCall().getNodeWithSimpleName().getName().asString()))
                            .toList();
                    for (MethodCallExpr addCall : allAddCalls) {
                        int addCallIndex = JavaRewriterUtil.findBlockStatementIndex(addCall);
                        if (addCallIndex >= attachCallIndex && addCallIndex < finalAddLocation) {
                            Statement statement = JavaRewriterUtil.findAncestorOrThrow(addCall, Statement.class);
                            statement.remove();
                            insertBlock.addStatement(finalAddLocation - 1, statement);
                        }
                    }
                }

            } else {
                throw new IllegalArgumentException("Reference component not found in the add call");
            }
        } else {
            throw new IllegalArgumentException(UNKNOWN_WHERE + where);
        }
    }

    public void duplicate(ComponentInfo component) {
        duplicate(component, true);
    }

    /**
     * Duplicates a component in the source code.
     *
     * @param component
     *            the component to duplicate
     * @param handleAdd
     *            true to automatically add the new component next to the old one,
     *            false to handle add calls like any other method call
     * @return a map from old component name to new component name
     */
    public DuplicateInfo duplicate(ComponentInfo component, boolean handleAdd) {
        if (component.routeConstructor() != null) {
            throw new IllegalArgumentException("Cannot duplicate a route class");
        }

        // Insert new component after the original component
        InsertionPoint insertionPoint = findInsertionPointForAppend(component);
        String oldName;
        List<MethodCallExpr> childAddCalls = new ArrayList<>();
        VariableDeclarationExpr variableDeclaration = null;
        AssignExpr assignExpr = null;

        String duplicatedName = null;
        if (component.localVariableName() != null) {
            oldName = component.localVariableName();
            duplicatedName = JavaRewriterUtil.findFreeVariableName(component.localVariableName(),
                    insertionPoint.getBlock());
            VariableDeclarator newLocalVariable = JavaRewriterUtil.clone(component.localVariableDeclarator());
            newLocalVariable.setName(duplicatedName);
            variableDeclaration = new VariableDeclarationExpr(newLocalVariable);
            JavaRewriterUtil.appendExpressionAsNextSiblingInBlockAncestor(component.localVariableDeclarator(),
                    new ExpressionStmt(variableDeclaration));
        } else if (component.fieldName() != null) {
            oldName = component.fieldName();
            duplicatedName = insertionPoint.getFreeVariableName(component.fieldName());
            FieldDeclaration newField;
            if (component.fieldDeclarationAndAssignment() != null) {
                newField = JavaRewriterUtil.clone(component.fieldDeclarationAndAssignment());
                if (newField.getVariables().isNonEmpty()) {
                    variableDeclaration = new VariableDeclarationExpr(newField.getVariable(0));
                }
            } else {
                newField = JavaRewriterUtil.clone(component.fieldDeclaration());
                if (newField.getVariables().size() > 1) {
                    newField.getVariables()
                            .removeIf(variableDeclarator -> !variableDeclarator.getNameAsString().equals(oldName));
                }
                // Assignment, e.g. button = new Button();
                assignExpr = JavaRewriterUtil.clone(component.assignmentExpression());
                assignExpr.setTarget(new NameExpr(duplicatedName));
                JavaRewriterUtil.appendExpressionAsNextSiblingInBlockAncestor(component.assignmentExpression(),
                        new ExpressionStmt(assignExpr));
            }
            newField.getVariable(0).setName(duplicatedName);
            JavaRewriterUtil.addFieldAfter(newField, component.fieldDeclaration());
        } else {
            // Inline
            oldName = null;
        }

        // Duplicate method calls
        if (duplicatedName != null) {
            List<MethodCallExpr> calls = JavaRewriterUtil.findMethodCallStatements(component);
            for (MethodCallExpr call : calls) {
                MethodCallExpr newCall = JavaRewriterUtil.clone(call);
                JavaRewriterUtil.setNameExprScope(newCall, new NameExpr(duplicatedName));
                insertionPoint.add(new ExpressionStmt(newCall));
                if (newCall.getNameAsString().equals("add")) {
                    childAddCalls.add(newCall);
                }
            }

            List<Expression> parameterUsages = JavaRewriterUtil.findParameterUsage(component);
            // excluding attach call
            Optional<Range> attachCallRangeOptional = component.attachCall().getNodeWithRange().getRange();
            if (attachCallRangeOptional.isPresent()) {
                Range attachRange = attachCallRangeOptional.get();
                parameterUsages = parameterUsages.stream().filter(parameter -> parameter.getRange().isPresent()
                        && !parameter.getRange().get().overlapsWith(attachRange)).toList();
            }
            for (Expression parameterUsage : parameterUsages) {
                MethodCallExpr methodCallExpr = JavaRewriterUtil.findAncestorOrThrow(parameterUsage,
                        MethodCallExpr.class);
                int argumentPosition = methodCallExpr.getArgumentPosition(parameterUsage);
                boolean addAsArg = false;
                Optional<Expression> scope = JavaRewriterUtil.getScopeIgnoreComposite(component, methodCallExpr);
                if (scope.isEmpty() || scope.get().isThisExpr()) {
                    Optional<String> classNameOpt = JavaRewriterUtil
                            .findAncestor(parameterUsage, ClassOrInterfaceDeclaration.class).getFullyQualifiedName();
                    if (classNameOpt.isEmpty()) {
                        throw new IllegalArgumentException("Could not find source class");
                    }
                    addAsArg = JavaRewriterUtil.isArrayArgument(classNameOpt.get(), methodCallExpr.getNameAsString(),
                            argumentPosition);
                }
                if (addAsArg) {
                    methodCallExpr.getArguments().add(argumentPosition + 1, new NameExpr(duplicatedName));
                } else {
                    MethodCallExpr clone = JavaRewriterUtil.clone(methodCallExpr);
                    clone.setArgument(argumentPosition, new NameExpr(duplicatedName));
                    insertionPoint.add(new ExpressionStmt(clone));
                }
            }
        }
        // Attach the new component
        if (handleAdd) {
            Expression componentAddArgument = JavaRewriterUtil.getAttachArgumentOrThrow(component);

            int addIndex;
            NodeWithArguments<?> expressionToAdd;
            Optional<Expression> parentNode = componentAddArgument.getParentNode().map(Expression.class::cast);
            if (parentNode.isPresent() && parentNode.get().isObjectCreationExpr()) {
                ObjectCreationExpr objectCreationExpr = parentNode.get().asObjectCreationExpr();
                addIndex = objectCreationExpr.getArguments().indexOf(componentAddArgument) + 1;
                expressionToAdd = objectCreationExpr;
            } else {
                addIndex = component.attachCall().getNodeWithArguments().getArgumentPosition(componentAddArgument) + 1;
                expressionToAdd = component.attachCall().getNodeWithArguments();
            }
            if (duplicatedName != null) {
                expressionToAdd.getArguments().add(addIndex, new NameExpr(duplicatedName));
            } else {
                expressionToAdd.getArguments().add(addIndex, JavaRewriterUtil.clone(component.objectCreationExpr()));
            }
        }

        Map<String, String> nameMappings;
        if (oldName != null) {
            nameMappings = Collections.singletonMap(oldName, duplicatedName);
        } else {
            nameMappings = Collections.emptyMap();
        }

        return new DuplicateInfo(nameMappings, childAddCalls, variableDeclaration, assignExpr);
    }

    /**
     * Adds the given code snippet to the source code either before the reference
     * component (Where.BEFORE) or by appending to the layout (Where.APPEND).
     *
     * @param referenceComponent
     *            the reference component (BEFORE) or container (APPEND) to add the
     *            code
     * @param where
     *            where to add the code
     * @param template
     *            the code to add, as JSON array of objects with "tag", "props" and
     *            "children"
     * @param options
     *            options that control how the template is added
     */
    public void addComponentUsingTemplate(ComponentInfo referenceComponent, Where where, List<JavaComponent> template,
            AddTemplateOptions options) {

        InsertionPoint insertionPoint = findInsertionPointForAppend(referenceComponent, where);
        if (where == Where.APPEND) {
            //
            // HorizontalLayout layout = new HorizontalLayout(button);
            // layout.setThis();
            // layout.addThat();
            // ...
            // add(layout);
            if (referenceComponent.routeConstructor() != null) {
                createComponentStatements(insertionPoint, null, template, "this", null, options);
            } else {
                createComponentStatements(insertionPoint, null, template,
                        JavaRewriterUtil.getFieldOrVariableName(referenceComponent), null, options);
            }

        } else if (where == Where.BEFORE) {
            // HorizontalLayout hl = new HorizontalLayout();
            // button = new Button("Hello");
            // button.setThis();
            // button.addThat();
            // hl.add(button);
            //
            // Insert constructor + setters before the reference constructor and
            // add the component so that it is added right before the reference
            createComponentStatements(insertionPoint, null, template,
                    JavaRewriterUtil.getFieldOrVariableName(referenceComponent), referenceComponent, options);

        } else {
            throw new IllegalArgumentException(UNKNOWN_WHERE + where);
        }
    }

    private InsertionPoint findInsertionPointForAppend(ComponentInfo component) {
        List<MethodCallExpr> functionCalls = JavaRewriterUtil.findMethodCallStatements(component);
        if (!functionCalls.isEmpty()) {
            // There are some component.setThis() or component.somethingElse()
            // calls, add a new component after those
            MethodCallExpr lastCall = functionCalls.get(functionCalls.size() - 1);
            return JavaRewriterUtil.findLocationAfter(lastCall);
        } else if (component.routeConstructor() != null) {
            return JavaRewriterUtil.findLocationAtEnd(component.routeConstructor().getBody());
        }

        // There are no component.anything() calls, add before the
        // add(component) call
        return JavaRewriterUtil.findLocationBefore(component.attachCall().expression());
    }

    public InsertionPoint findInsertionPointForAppend(ComponentInfo component, Where where) {
        if (where == Where.APPEND) {
            return findInsertionPointForAppend(component);
        } else if (where == Where.BEFORE) {
            BlockStmt insertBlock = component.componentCreateScope();
            int insertIndex = JavaRewriterUtil.findBlockStatementIndex(component.objectCreationExpr());
            return new InsertionPoint(insertBlock, insertIndex);

        } else {
            throw new IllegalArgumentException(UNKNOWN_WHERE + where);

        }
    }

    public void setAlignment(ComponentInfo component, String alignItemsClassName, String justifyContentClassName) {
        LumoRewriterUtil.removeClassNameArgs(component, "AlignItems", "JustifyContent");
        LumoRewriterUtil.addLumoUtilityImport(compilationUnit);
        if (alignItemsClassName != null) {
            addOrReplaceLumoClass(component, "AlignItems", alignItemsClassName);
        }
        if (justifyContentClassName != null) {
            addOrReplaceLumoClass(component, "JustifyContent", justifyContentClassName);
        }
    }

    /**
     * Sets gap to selected component
     *
     * @param component
     *            component to set gap
     * @param lumoClassAll
     *            gap all value. e.g. gap-m. might be <code>null</code>
     * @param lumoClassColumn
     *            gap column value which starts with gap-x prefix, e.g. gap-x-xs.
     *            might be <code>null</code>
     * @param lumoClassRow
     *            gap row value which starts with gap-y prefix, e.g. gap-y-xs. might
     *            be <code>null</code>
     */
    public void setGap(ComponentInfo component, String lumoClassAll, String lumoClassColumn, String lumoClassRow) {
        LumoRewriterUtil.removeClassNameArgs(component, "Gap", "Gap.Column", "Gap.Row");
        LumoRewriterUtil.addLumoUtilityImport(compilationUnit);
        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil.findMethodCallStatements(component);
        methodCallStatements.stream()
                .filter(methodCallExpr -> StringUtils.equals(methodCallExpr.getNameAsString(), SET_SPACING_METHOD))
                .findAny().ifPresent(JavaRewriterUtil::removeStatement);
        LumoRewriterUtil.removeThemeArgStartsWith(methodCallStatements, "spacing");

        if (lumoClassAll != null) {
            addOrReplaceLumoClass(component, "Gap", lumoClassAll);
        }
        if (lumoClassRow != null) {
            addOrReplaceLumoClass(component, "Gap.Row", lumoClassRow);
        }
        if (lumoClassColumn != null) {
            addOrReplaceLumoClass(component, "Gap.Column", lumoClassColumn);
        }
        if (lumoClassAll == null && lumoClassColumn == null && lumoClassRow == null) {
            MethodCallExpr call = JavaRewriterUtil.addAfterLastFunctionCall(methodCallStatements, SET_SPACING_METHOD,
                    new BooleanLiteralExpr(false));
            if (call == null) {
                replaceOrAddCall(component, SET_SPACING_METHOD, false);
            }
        }
    }

    private void addOrReplaceLumoClass(ComponentInfo component, String lumoUtilityClassName, String cssClassName) {
        List<Expression> gapExpressions = LumoRewriterUtil.getLumoMethodArgExpressions(List.of(lumoUtilityClassName),
                List.of(cssClassName));
        boolean added = LumoRewriterUtil.addClassNameWithArgs(component, gapExpressions);
        if (!added) {
            replaceOrAddCall(component, ADD_CLASS_NAMES_METHOD, gapExpressions.toArray(new Object[0]));
        }
    }

    public void setPadding(ComponentInfo component, String all, String top, String right, String bottom, String left) {
        // removing all padding classes
        String[] lumoPaddingInnerClassNames = new String[] { "Padding", "Padding.Bottom", "Padding.End",
                "Padding.Horizontal", "Padding.Left", "Padding.Right", "Padding.Start", "Padding.Top",
                "Padding.Vertical" };

        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil.findMethodCallStatements(component);
        methodCallStatements.stream()
                .filter(methodCallExpr -> StringUtils.equals(methodCallExpr.getNameAsString(), "setPadding"))
                .forEach(JavaRewriterUtil::removeStatement);
        LumoRewriterUtil.removeSetThemeArgs(methodCallStatements, "padding");

        LumoRewriterUtil.removeClassNameArgs(component, lumoPaddingInnerClassNames);
        LumoRewriterUtil.addLumoUtilityImport(compilationUnit);

        if (all != null) {
            addOrReplaceLumoClass(component, "Padding", all);
        }
        if (top != null) {
            addOrReplaceLumoClass(component, "Padding.Top", top);
        }
        if (right != null) {
            addOrReplaceLumoClass(component, "Padding.Right", right);
        }
        if (bottom != null) {
            addOrReplaceLumoClass(component, "Padding.Bottom", bottom);
        }
        if (left != null) {
            addOrReplaceLumoClass(component, "Padding.Left", left);
        }
    }

    /**
     * Merges all the components and wraps them using the given component and places
     * the result in place of the first component.
     *
     * @param components
     *            The components to merge. The first component will be replaced with
     *            the wrapper component
     * @param wrapperComponent
     *            The component to wrap the merged components in.
     */
    public void mergeAndReplace(List<ComponentInfo> components, JavaComponent wrapperComponent) {
        ComponentInfo firstComponent = components.get(0);
        BlockStmt firstComponentAttachScope = firstComponent.componentAttachScope()
                .orElseThrow(() -> new IllegalArgumentException("Routes cannot be wrapped"));
        for (int i = 1; i < components.size(); i++) {
            ComponentInfo component = components.get(i);
            Optional<BlockStmt> componentAttachScope = component.componentAttachScope();
            if (componentAttachScope.isEmpty() || componentAttachScope.get() != firstComponentAttachScope) {
                throw new IllegalArgumentException(
                        "Only components attached in the same block are currently supported");
            }
        }

        // Add wrapper init code where the first component is attached
        int wrapperInsertIndex = JavaRewriterUtil.findBlockStatementIndex(firstComponent.attachCall().getNode());
        InsertionPoint firstComponentAttachLocation = new InsertionPoint(firstComponentAttachScope, wrapperInsertIndex);

        List<VariableDeclarator> statements = createComponentStatements(firstComponentAttachLocation, null,
                wrapperComponent, false, null, null, new AddTemplateOptions(false));

        // Remove all attach calls for the components and attach them to the
        // wrapper instead. Replace the first component add with an add for the
        // wrapper
        NameExpr wrapperVariable = new NameExpr(statements.get(0).getNameAsString());
        MethodCallExpr addCall = new MethodCallExpr(wrapperVariable, "add");

        Node wrapperAdd = null;
        for (int i = 0; i < components.size(); i++) {
            ComponentInfo component = components.get(i);
            Optional<Expression> componentAddArgument;
            if (i == 0) {
                componentAddArgument = Optional.of(JavaRewriterUtil.getAttachArgumentOrThrow(component));
                // Attach wrapper in place of the first component
                Optional<Node> maybeParent = componentAddArgument.flatMap(Node::getParentNode);
                if (maybeParent.isEmpty()) {
                    throw new IllegalArgumentException("No add argument found for the first component");
                }

                wrapperAdd = maybeParent.get();
                wrapperAdd.replace(componentAddArgument.get(), wrapperVariable);
            } else {
                componentAddArgument = removeAttachCall(component);
            }
            componentAddArgument.ifPresent(addCall::addArgument);
        }
        // Add the wrapper.add(components) call
        if (wrapperAdd == null) {
            throw new IllegalArgumentException("Wrapper was never added");
        }
        firstComponentAttachLocation.getBlock().addStatement(JavaRewriterUtil.findBlockStatementIndex(wrapperAdd),
                addCall);
    }

    /**
     * Replaces a parameter name in a method call.
     *
     * @param call
     * @param oldVariableName
     * @param newVariableName
     */
    public void replaceCallParameter(NodeWithArguments<?> call, String oldVariableName, String newVariableName) {
        call.getArguments().stream().forEach(arg -> {
            if (arg.isNameExpr() && arg.asNameExpr().getNameAsString().equals(oldVariableName)) {
                arg.asNameExpr().setName(newVariableName);
            }
        });
    }

    public void createComponentStatements(InsertionPoint insertionPoint, JavaComponent parent,
            List<JavaComponent> template, String layoutVariableName, ComponentInfo referenceComponent,
            AddTemplateOptions options) {
        for (JavaComponent javaComponent : template) {
            Optional<CustomComponentHandle> customComponentHandleOpt = CustomComponentHandler.get(javaComponent);
            if (customComponentHandleOpt.isPresent()) {
                customComponentHandleOpt.get().createComponentStatements(this, javaComponent, insertionPoint, parent,
                        layoutVariableName, referenceComponent, options);
            } else {
                createComponentStatements(insertionPoint, parent, javaComponent, true, layoutVariableName,
                        referenceComponent, options);
            }
        }
    }

    public List<VariableDeclarator> createComponentStatements(InsertionPoint insertionPoint, JavaComponent parent,
            JavaComponent maybeJavaComponent, boolean attach, String layoutVariableName,
            ComponentInfo referenceComponent, AddTemplateOptions options) {

        final JavaComponent javaComponent;
        List<VariableDeclarator> createdVariables = new ArrayList<>();

        if ("Message".equals(maybeJavaComponent.tag()) || "Item".equals(maybeJavaComponent.tag())
                || "RadioButton".equals(maybeJavaComponent.tag()) || "CustomField".equals(maybeJavaComponent.tag())) {
            // A custom field is not supported right now as it needs a custom
            // class
            // vaadin-message has no Flow component
            // vaadin-radio-button has no Flow component
            // VaadinItem cannot be used without listbox
            return createdVariables;
        }
        String componentClassName = maybeJavaComponent.className() != null ? maybeJavaComponent.className()
                : FlowComponentQuirks.getClassForComponent(maybeJavaComponent);
        if (componentClassName.equals("com.vaadin.flow.component.treegrid.TreeGrid")) {
            // React does not differ between Grid and TreeGrid
            javaComponent = maybeJavaComponent.withTag("TreeGrid");
        } else {
            javaComponent = maybeJavaComponent;
        }

        ClassOrInterfaceType fullType = StaticJavaParser.parseClassOrInterfaceType(componentClassName);

        Map<String, Object> setters = new HashMap<>();
        Class<?> componentType = JavaRewriterUtil.getClass(fullType.getNameWithScope());

        // Import
        JavaRewriterUtil.addImport(compilationUnit, fullType.getNameWithScope());
        ClassOrInterfaceType variableType = JavaRewriterUtil.clone(fullType).removeScope();
        ClassOrInterfaceType itemsType = null;
        String itemsTypeName = javaComponent.metadata().getItemType();

        if (itemsTypeName != null) {
            itemsType = JavaRewriterUtil.clone(StaticJavaParser.parseClassOrInterfaceType(itemsTypeName)).removeScope();
        }
        // we create a second type for instantiation to avoid explicit type and
        // allow to modify separately the ClassOrInterfaceType for declaration
        // and definition
        ClassOrInterfaceType typeForInstantiation = JavaRewriterUtil.clone(variableType);

        // Find data entity available name in case of data provider
        String dataEntityRecordName = JavaRewriterUtil.findFreeRecordName(DEFAULT_ENTITY_RECORD_NAME,
                JavaRewriterUtil.findAncestor(insertionPoint.getBlock(), ClassOrInterfaceDeclaration.class));

        // Constructor call
        ObjectCreationExpr initializer = JavaRewriterUtil.createComponentConstructor(javaComponent,
                typeForInstantiation, dataEntityRecordName,
                (String neededImport) -> JavaRewriterUtil.addImport(compilationUnit, neededImport));
        if (itemsType != null) {
            initializer.getType().setTypeArguments(JavaRewriterUtil.createEmptyType());
            JavaRewriterUtil.addImport(compilationUnit, itemsTypeName);
            variableType.setTypeArguments(itemsType);
        }
        // Gets a valid available variable name for the new component
        String variableName = JavaRewriterUtil.generateVariableName(javaComponent, variableType, insertionPoint);

        if (options.javaFieldsForLeafComponents() && javaComponent.children().isEmpty()) {
            ClassOrInterfaceDeclaration classType = JavaRewriterUtil.findAncestorOrThrow(insertionPoint.getBlock(),
                    ClassOrInterfaceDeclaration.class);
            FieldDeclaration fieldDeclaration = classType.addFieldWithInitializer(variableType, variableName,
                    initializer, Modifier.Keyword.PRIVATE, Modifier.Keyword.FINAL);
            // Move field before all constructors/methods
            JavaRewriterUtil.moveAboveMethodsAndConstructors(fieldDeclaration, classType);
        } else {
            VariableDeclarator decl = new VariableDeclarator(variableType, variableName, initializer);
            VariableDeclarationExpr declarationExpr = new VariableDeclarationExpr(decl);
            insertionPoint.add(new ExpressionStmt(declarationExpr));
            createdVariables.add(decl);
        }
        // A FieldAccessExpr always requires a scope, but we are verifying that the
        // field
        // name is unique, so we do not need that
        Expression variableReference = new NameExpr(variableName);
        JavaDataProviderHandler.FieldOrVariable fieldOrVariable = new JavaDataProviderHandler.FieldOrVariable(
                variableReference, variableType, initializer);

        // CHART HACK: THE CHART IS NOT VISIBLE BY DEFAULT BECAUSE OF ITS SIZE
        // SO WE SET A DEFAULT SIZE TO MAKE IT VISIBLE
        if (javaComponent.tag() != null && javaComponent.tag().equalsIgnoreCase("Chart")) {
            insertSetter(insertionPoint, fieldOrVariable.reference(), "setMinHeight", "400px", compilationUnit,
                    javaComponent);
        }

        // We add data items included as props when the item is set
        // with in a data provider bean like
        // ListDataView::setItems(T... items)
        // SO FIRST WE HANDLE PROPERTIES
        JavaDataProviderHandler.handleDataStatementsAndClearDataProps(compilationUnit, fieldOrVariable, javaComponent,
                insertionPoint, dataEntityRecordName);

        // Properties setters
        javaComponent.props().forEach((prop, value) -> {
            if (prop.startsWith("_")) {
                // These are used for passing parent props through a child, e.g. label for a tab
                // sheet tab content
                return;
            }
            if (FlowComponentQuirks.skipProps(javaComponent, prop)) {
                return;
            }
            SetterAndValue setterAndValue = JavaRewriterUtil.getSetterAndValue(componentType, prop, value);
            setters.put(setterAndValue.setter(), setterAndValue.value());
        });
        // Removes setter property to be used in constructor
        JavaRewriterUtil.getSingleStringParamConstructor(fullType, setters.keySet()).ifPresent(constructorProp -> {
            Object value = setters.remove(constructorProp);
            getParameterList(insertionPoint, value).forEach(initializer::addArgument);
        });

        for (Map.Entry<String, Object> setter : setters.entrySet()) {
            Object value = setter.getValue();
            if (value instanceof JavaComponent javaComponentValue) {
                value = createSubComponentStatements(insertionPoint, javaComponent, List.of(javaComponentValue),
                        options).get(0);
            } else if (setter.getKey().equalsIgnoreCase(ADD_ITEM_METHOD) && value instanceof List<?> list) {
                for (Object item : list) {
                    insertItemsPropToAddItem(compilationUnit, javaComponent, insertionPoint,
                            fieldOrVariable.reference(), null, setter.getKey(), item);
                }
                continue;
            }
            insertSetter(insertionPoint, fieldOrVariable.reference(), setter.getKey(), value, compilationUnit,
                    javaComponent);
        }

        // Child components
        // First we remove children that are not meant to be added as components
        // but as method calls
        // THEN WE MANAGE CHILDREN
        List<JavaComponent> methodChildren = FlowComponentQuirks.getMethodCallChildren(javaComponent);

        javaComponent.children().removeAll(methodChildren);

        // Then we add the children as method calls
        createMethodCallChildrenStatements(insertionPoint, javaComponent, methodChildren, fieldOrVariable.reference(),
                dataEntityRecordName);

        // Then we add the children as components when items are not
        // added with a data provider but as dedicated types of items like
        // MessageList::setItems(com.vaadin.flow.component.messages.MessageListItem...
        // items )
        List<JavaComponent> children = javaComponent.children().stream().map(child -> {
            if (FlowComponentQuirks.isTabSheetDefinition(child)) {
                JavaComponent tabContent;
                if (child.children().isEmpty()) {
                    // Use an empty div as there must be some content
                    Map<String, Object> emptyText = new HashMap<>();
                    emptyText.put("text", "");
                    tabContent = new JavaComponent("div", null, emptyText, Collections.emptyList());
                } else {
                    tabContent = child.children().get(0);
                }
                tabContent.props().put("_label", child.props().get("label"));
                return tabContent;
            }
            return child;
        }).toList();
        createComponentStatements(insertionPoint, javaComponent, children, variableName, null, options);

        if (attach) {
            attachComponent(insertionPoint, javaComponent, parent, layoutVariableName, referenceComponent,
                    fieldOrVariable.reference(), variableName);
        }

        return createdVariables;
    }

    public void attachComponent(InsertionPoint insertionPoint, JavaComponent component, JavaComponent parent,
            String layoutVariableName, ComponentInfo referenceComponent, Expression variableNameExpr,
            String variableName) {
        // Attach component
        if (component.tag() != null && component.tag().equalsIgnoreCase(CHART_SERIES_CLASS)) {
            // ChartSeries are added to the Chart, not to the layout
            MethodCallExpr getConfigurationCall = new MethodCallExpr(new NameExpr(layoutVariableName),
                    "getConfiguration", new NodeList<>());
            MethodCallExpr addSeriesCall = new MethodCallExpr(getConfigurationCall, "addSeries",
                    new NodeList<>(variableNameExpr));
            insertionPoint.add(new ExpressionStmt(addSeriesCall));
        } else if (referenceComponent != null) {
            AttachExpression referenceAttach = referenceComponent.attachCall();
            if (referenceAttach.getNodeWithArguments().getArguments().size() == 1) {
                // add(reference)
                ExpressionStmt addCall = createAddCall(component, parent,
                        referenceAttach.getNodeWithOptionalScope().getScope()
                                .orElse(JavaRewriterUtil.isNodeInCompositeClass(referenceAttach.getNode())
                                        ? new MethodCallExpr("getContent")
                                        : null),
                        variableNameExpr);
                BlockStmt block = JavaRewriterUtil.findAncestorOrThrow(referenceAttach.getNode(), BlockStmt.class);
                block.addStatement(JavaRewriterUtil.findBlockStatementIndex(referenceAttach.getNode()), addCall);
            } else {
                Expression referenceArgument;
                // add(..., reference,...)
                if (layoutVariableName == null) {
                    // Reference is an inline call, like add(new Button()
                    referenceArgument = referenceComponent.objectCreationExpr();
                } else {
                    referenceArgument = referenceAttach.getNodeWithArguments().getArguments().stream()
                            .filter(arg -> arg instanceof NameExpr nameExpr
                                    && nameExpr.getNameAsString().equals(layoutVariableName))
                            .findFirst().orElse(null);
                }
                if (referenceArgument == null) {
                    throw new IllegalArgumentException(
                            "Did not find reference argument ('" + variableName + "') in " + referenceAttach);
                }
                int index = referenceAttach.getNodeWithArguments().getArguments().indexOf(referenceArgument);
                referenceAttach.getNodeWithArguments().getArguments().add(index, variableNameExpr);
            }
        } else if (layoutVariableName != null) {
            Expression scope = null;
            if (!layoutVariableName.equals("this")) {
                scope = new NameExpr(layoutVariableName);
            } else if (JavaRewriterUtil.isNodeInCompositeClass(insertionPoint.getBlock())) {
                scope = new MethodCallExpr("getContent");
            }
            ExpressionStmt addCall = createAddCall(component, parent, scope, variableNameExpr);
            // Just adding to the given layout
            insertionPoint.add(addCall);
        } else {
            throw new IllegalArgumentException(
                    "Either layoutVariableName or referenceAttach must be given to attach a component");
        }
    }

    public NodeList<Expression> getParameterList(InsertionPoint insertionPoint, Object value) {
        if (value instanceof List<?> list) {
            if (list.isEmpty()) {
                return new NodeList<>();
            }
            if (list.get(0) instanceof JavaComponent) {
                return JavaRewriterUtil.toExpressionList(createSubComponentStatements(insertionPoint, null,
                        (List<JavaComponent>) list, new AddTemplateOptions(false)));
            }
        }
        return JavaRewriterUtil.toExpressionList(value);
    }

    /**
     * Insert setters for given java component. TODO move this class into
     * {@link CustomComponentHandle} and remove it from here
     *
     * @param insertionPoint
     *            Insertion point to add setters
     * @param owner
     *            owner of the setters
     * @param setterName
     *            setterName from JavaComponent props
     * @param value
     *            setterValue from JavaComponent props
     * @param compilationUnit
     *            Compilation unit of the file
     * @param javaComponent
     *            Java component itself
     */
    public void insertSetter(InsertionPoint insertionPoint, Expression owner, String setterName, Object value,
            CompilationUnit compilationUnit, JavaComponent javaComponent) {
        if (setterName.equals("setStyle")) {
            insertStyles(insertionPoint, owner, (Map<String, String>) value);
        } else if (setterName.equals("setClassName")) {
            // replace setClassName with addClassNames and use Lumo Variables if possible.
            Expression addClassNameExp = LumoRewriterUtil.createAddClassNameExprUsingLumoVariables(owner, value,
                    compilationUnit);
            if (addClassNameExp == null) {
                NodeList<Expression> parameterExpression = getParameterList(insertionPoint, value);
                addClassNameExp = new MethodCallExpr(owner, setterName, parameterExpression);
            }
            insertionPoint.add(new ExpressionStmt(addClassNameExp));
        } else if (setterName.equals("setSlot")) {
            // getElement().setAttribute("slot", value)
            MethodCallExpr getElement = new MethodCallExpr(owner, "getElement");
            MethodCallExpr setAttributeCall = new MethodCallExpr(getElement, "setAttribute",
                    new NodeList<>(JavaRewriterUtil.toExpression("slot"), JavaRewriterUtil.toExpression(value)));
            insertionPoint.add(new ExpressionStmt(setAttributeCall));
        } else if (javaComponent.tag() != null && javaComponent.tag().equalsIgnoreCase("Chart")
                && !setterName.equals("setMinHeight")) {
            if (setterName.equals("setAdditionalOptions")) {
                Map<String, Object> additionalOptions = (Map<String, Object>) value;
                for (Map.Entry<String, Object> entry : additionalOptions.entrySet()) {
                    String key = entry.getKey();
                    Object val = entry.getValue();
                    if (key.equalsIgnoreCase("xAxis") || key.equalsIgnoreCase("yAxis")) {
                        Map<String, Object> axis = (Map<String, Object>) val;
                        for (Map.Entry<String, Object> entryAxis : axis.entrySet()) {
                            String keyAxis = entryAxis.getKey();
                            Object valAxis = entryAxis.getValue();
                            if (keyAxis.equalsIgnoreCase("categories")) {
                                insertionPoint.add(new ExpressionStmt(FlowComponentQuirks.getPropertySetExpression(
                                        compilationUnit, javaComponent, key, keyAxis, valAxis, owner, insertionPoint)));
                            } else if (keyAxis.equalsIgnoreCase("title")) {
                                insertionPoint.add(new ExpressionStmt(FlowComponentQuirks.getPropertySetExpression(
                                        compilationUnit, javaComponent, key, keyAxis, valAxis, owner, insertionPoint)));
                            }
                        }
                    }
                }
            } else {
                NodeList<Expression> parameterExpression = getParameterList(insertionPoint, value);
                MethodCallExpr setterCall = new MethodCallExpr(
                        FlowComponentQuirks.getPropertySetExpression(compilationUnit, javaComponent, null, setterName,
                                value, owner, insertionPoint),
                        setterName, parameterExpression);
                insertionPoint.add(new ExpressionStmt(setterCall));
            }
        } else if (javaComponent.tag() != null && javaComponent.tag().equalsIgnoreCase(CHART_SERIES_CLASS)
                && setterName.equals("setPlotOptions")) {
            insertionPoint.add(new ExpressionStmt(FlowComponentQuirks.getPropertySetExpression(compilationUnit,
                    javaComponent, null, setterName, value, owner, insertionPoint)));
        } else {
            NodeList<Expression> parameterExpression = getParameterList(insertionPoint, value);
            if (CHART_SERIES_CLASS.equals(javaComponent.tag()) && "setData".equals(setterName)
                    && value instanceof List<?> list && !(list).isEmpty() && (list).get(0) instanceof JavaComponent) {
                // We need to wrap to a List only if the data is DataSeriesItem
                // instances because
                // DataSeries has
                // - setData(Number... values)
                // - setData(List<DataSeriesItem> data)
                parameterExpression = wrapWithListOf(parameterExpression);
                JavaRewriterUtil.addImport(compilationUnit, "java.util.List");
            }
            MethodCallExpr setterCall = new MethodCallExpr(owner, setterName, parameterExpression);
            insertionPoint.add(new ExpressionStmt(setterCall));
            if (value instanceof Enum<?>) {
                JavaRewriterUtil.addImport(compilationUnit, value.getClass().getCanonicalName());
            }
        }
    }

    private NodeList<Expression> wrapWithListOf(NodeList<Expression> parameterExpression) {
        return new NodeList<>(new MethodCallExpr(new NameExpr("List"), "of", parameterExpression));
    }

    private void insertStyles(InsertionPoint insertionPoint, Expression owner, Map<String, String> styles) {
        // button.getStyle().set("p1","value1").set("p2","value2");
        MethodCallExpr finalCall = new MethodCallExpr(owner, "getStyle");
        for (Map.Entry<String, String> entry : styles.entrySet()) {
            finalCall = new MethodCallExpr(finalCall, "set", new NodeList<>(
                    JavaRewriterUtil.toExpression(entry.getKey()), JavaRewriterUtil.toExpression(entry.getValue())));
        }
        insertionPoint.add(new ExpressionStmt(finalCall));
    }

    private NodeList<Expression> createSubComponentStatements(InsertionPoint insertionPoint, JavaComponent parent,
            List<JavaComponent> components, AddTemplateOptions options) {
        List<VariableDeclarator> variables = new ArrayList<>();
        for (JavaComponent javaComponent : components) {
            variables.addAll(
                    createComponentStatements(insertionPoint, parent, javaComponent, false, null, null, options));
        }
        return new NodeList<>(
                variables.stream().map(VariableDeclarator::getName).map(NameExpr::new).toArray(NameExpr[]::new));
    }

    private ExpressionStmt createAddCall(JavaComponent component, JavaComponent parent, Expression scope,
            Expression toAdd) {
        String methodName = "add";
        NodeList<Expression> parameters = new NodeList<>(toAdd);
        if (parent != null) {
            if (parent.tag().equals("SideNav") || parent.tag().equals("SideNavItem")) {
                methodName = ADD_ITEM_METHOD;
            } else if (parent.tag().equals("TabSheet")) {
                parameters.add(0, new StringLiteralExpr((String) component.props().get("_label")));
            }
        }
        return new ExpressionStmt(new MethodCallExpr(scope, methodName, parameters));
    }

    /**
     * Returns the source code.
     *
     * @return the source code
     */
    protected String getSource() {
        return source;
    }

    /**
     * This method is used to create method call statements for the children of a
     * component when the mapping from React template to Java code is <br>
     * React children -> Flow method calls.<br>
     * This mapping is quite adhoc per component and should be defined in the
     * {@link FlowComponentQuirks} utility class.
     */
    private void createMethodCallChildrenStatements(InsertionPoint insertionPoint, JavaComponent parent,
            List<JavaComponent> methodChildren, Expression owner, String dataEntityRecordName) {
        List<JavaComponent> gridColumns = new ArrayList<>();
        List<JavaComponent> gridTreeColumns = new ArrayList<>();

        for (JavaComponent child : methodChildren) {
            if (FlowComponentQuirks.isGridTreeColumnDefinition(child)) {
                gridTreeColumns.add(child);
            }
            if (FlowComponentQuirks.isGridColumnDefinition(child)) {
                gridColumns.add(child);
            } else {
                // Get the method call expression for the child component (if
                // any
                List<MethodCallExpr> expression = FlowComponentQuirks.getMethodCallExprFromComponent(compilationUnit,
                        child, owner, dataEntityRecordName);
                for (MethodCallExpr methodCallExpr : expression) {
                    insertionPoint.add(new ExpressionStmt(methodCallExpr));
                }
            }
        }
        if (!gridColumns.isEmpty()) {
            // We use the items to filter columns that are not supported because of the type
            // of the data
            List<Map<String, Object>> itemsFromProperty = parent.getItemsFromProperty();
            List<String> supportedColumnsKeys = itemsFromProperty.stream().flatMap(map -> map.keySet().stream())
                    .collect(Collectors.toSet()).stream().collect(Collectors.toList());
            // We skip duplicate columns
            List<String> addedColumns = new ArrayList<>();
            MethodCallExpr expression = new MethodCallExpr(owner, "setColumns");
            for (JavaComponent gridColumn : gridColumns) {
                String columnName = gridColumn.props().get("path").toString();
                if (FlowComponentQuirks.isGridColumnDefinition(gridColumn) && supportedColumnsKeys.contains(columnName)
                        && !addedColumns.contains(columnName)) {
                    expression.addArgument(new StringLiteralExpr(columnName));
                    addedColumns.add(columnName);
                }
            }
            insertionPoint.add(new ExpressionStmt(expression));
        }
        if (!gridTreeColumns.isEmpty()) {
            MethodCallExpr expression = new MethodCallExpr(owner, "setHierarchyColumn");
            for (JavaComponent gridTreeColumn : gridTreeColumns) {
                expression.addArgument(new StringLiteralExpr(gridTreeColumn.props().get("path").toString()));
            }
            insertionPoint.add(new ExpressionStmt(expression));
        }
    }

    private void insertItemsPropToAddItem(CompilationUnit compilationUnit, JavaComponent javaComponent,
            InsertionPoint insertionPoint, Expression owner, Expression parent, String setterName, Object value) {
        if (javaComponent.tag().equals("MenuBar") && setterName.equals(ADD_ITEM_METHOD)) {
            JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.component.contextmenu.MenuItem");
            JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.component.contextmenu.SubMenu");
            FlowComponentQuirks.menuBarInsertItemsPropsToAddItem(javaComponent, insertionPoint, owner, parent,
                    setterName, value, false);
        } else {
            throw new IllegalArgumentException("Invalid or not supported items to be added separately for "
                    + javaComponent.tag() + " and setter " + setterName + " with value " + value.getClass().getName());
        }
    }

    /**
     * Adds a comment to the specified component in the source code.
     *
     * @param componentInfo
     *            the component to which the comment should be added
     * @param comment
     *            the comment to add
     * @param commentType
     *            the type of comment (LINE, BLOCK, JAVADOC)
     */
    public void addComment(ComponentInfo componentInfo, String comment, CommentType commentType) {
        Node targetNode = getNodeForAddingComment(componentInfo);

        if (targetNode != null) {
            Comment newComment = switch (commentType) {
            case LINE -> new LineComment(comment);
            case BLOCK -> new BlockComment(comment);
            };
            targetNode.setComment(newComment);
        } else {
            throw new IllegalArgumentException("Unable to determine the target node for the component.");
        }
    }

    /**
     * Helper method to get the appropriate Node from a ComponentInfo.
     *
     * @param componentInfo
     *            the component information
     * @return the corresponding Node, or null if not found
     */
    private Node getNodeForAddingComment(ComponentInfo componentInfo) {
        if (componentInfo.localVariableDeclarator() != null) {
            return componentInfo.localVariableDeclarator().getParentNode().orElse(null);
        } else if (componentInfo.assignmentExpression() != null) {
            return componentInfo.assignmentExpression();
        } else if (componentInfo.fieldDeclaration() != null) {
            return componentInfo.fieldDeclaration();
        } else if (componentInfo.fieldDeclarationAndAssignment() != null) {
            return componentInfo.fieldDeclarationAndAssignment();
        } else if (componentInfo.routeConstructor() != null) {
            return componentInfo.routeConstructor();
        } else if (componentInfo.attachCall() != null) {
            return componentInfo.attachCall().getNode();
        }
        return null;
    }

    /**
     * Enum representing the type of comment to be added.
     */
    public enum CommentType {
        LINE, BLOCK
    }
}
