package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.vaadin.base.devserver.FileWatcher;
import com.vaadin.copilot.ProjectManager;
import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;
import com.vaadin.flow.function.SerializableConsumer;
import com.vaadin.flow.internal.DevModeHandlerManager;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Watches target and source files to detect compilation required cases.
 */
public class SourceSyncChecker {

    private static final String JAVA_FILE_EXTENSION = ".java";

    private final Set<File> updatedAtLeastOnce = new HashSet<>();
    private final Set<String> notSyncIssuesStacktrace = new HashSet<>();

    private final ProjectManager projectManager;

    private final SerializableConsumer<File> sourceClassChangeListener = sourceFile -> {
        if (sourceFile.isFile() && sourceFile.getAbsolutePath().endsWith(JAVA_FILE_EXTENSION)) {
            updatedAtLeastOnce.add(sourceFile);
        }
    };
    private final List<FileWatcher> sourceFileWatchers = new ArrayList<>();

    /**
     * Starts the source file watcher with the given application configuration.
     *
     * @param projectManager
     *            project manager for getting the application configuration
     * @param devModeHandler
     *            flow dev mode handler for registering file watcher shut down
     */
    public SourceSyncChecker(ProjectManager projectManager, DevModeHandlerManager devModeHandler) {
        this.projectManager = projectManager;
        startSourceFileWatchers(devModeHandler);
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SourceSyncChecker.class);
    }

    /**
     * Starts a watcher to watch .java file changes in order to collect which files
     * may be out of synced when an exception occurs.
     *
     * @param devModeHandler
     *            flow dev mode handler for registering file watcher shut down
     */
    private void startSourceFileWatchers(DevModeHandlerManager devModeHandler) {
        List<FileWatcher> watchers = projectManager.getSourceFolders().stream().map(this::startSourceFileWatcher)
                .filter(Objects::nonNull).toList();
        sourceFileWatchers.addAll(watchers);
        registerWatcherShutdownCommand(devModeHandler);
    }

    private FileWatcher startSourceFileWatcher(Path sourcePath) {
        try {
            File sourceFolder = sourcePath.toFile();
            FileWatcher sourceFileWatcher = new FileWatcher(sourceClassChangeListener, sourceFolder);
            sourceFileWatcher.start();
            getLogger().debug("Started watching {}", sourceFolder);
            return sourceFileWatcher;
        } catch (IOException e) {
            getLogger().error("Could not start file watched for source classes", e);
        }
        return null;
    }

    private void registerWatcherShutdownCommand(DevModeHandlerManager devModeHandlerManager) {
        devModeHandlerManager.registerShutdownCommand(() -> {
            for (FileWatcher sourceFileWatcher : sourceFileWatchers) {
                try {
                    sourceFileWatcher.stop();
                } catch (IOException e) {
                    getLogger().error("Could not stop file watcher for source classes", e);
                }
            }
            sourceFileWatchers.clear();
        });
    }

    /**
     * Looks for file has been updated since the startup. Additionally, caches
     * exceptions that happens without any files changes.
     *
     * @param exception
     *            Exception that happens when finding component info fails.
     * @return true if file may be out of sync, false otherwise
     */
    public boolean maybeOutOfSync(ComponentInfoNotFoundException exception) {
        if (notSyncIssuesStacktrace.contains(ExceptionUtils.getStackTrace(exception))) {
            return false;
        }
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = exception.getComponentTypeAndSourceLocation();
        Optional<File> maybeSourceFile = componentTypeAndSourceLocation.javaFile();
        if (maybeSourceFile.isEmpty()) {
            return false;
        }
        File sourceFile = maybeSourceFile.get();
        if (!updatedAtLeastOnce.contains(sourceFile)) {
            notSyncIssuesStacktrace.add(ExceptionUtils.getStackTrace(exception));
            return false;
        }
        return updatedAtLeastOnce.contains(sourceFile);
    }

    List<FileWatcher> getSourceFileWatchers() {
        return sourceFileWatchers;
    }

}
