package com.vaadin.copilot;

import java.lang.reflect.Method;
import java.util.List;

/**
 * Utility class for connecting to a service method.
 */
public class ConnectToService {

    private static final String PAGEABLE = "org.springframework.data.domain.Pageable";
    private static final String FLOW_FILTER = "com.vaadin.flow.spring.data.filter.Filter";
    private static final String HILLA_FILTER = "com.vaadin.hilla.crud.filter.Filter";

    /**
     * The type of binding that can be used for a Flow Grid.
     */
    public enum BindingType {
        EAGER, LAZY, NONE
    }

    /**
     * Checks if a Flow Grid can be bound to the given service method.
     *
     * @param serviceMethod
     *            the service method to check
     * @return the binding type that can be used
     */
    public static BindingType canBindFlowGridToService(SpringBridge.ServiceMethodInfo serviceMethod) {

        Method method = serviceMethod.serviceMethod();
        if (method.getReturnType() != List.class) {
            return BindingType.NONE;
        }

        Class<?>[] parameterTypes = method.getParameterTypes();
        if (parameterTypes.length == 0) {
            // Eager binding is OK
            return BindingType.EAGER;
        }

        if (parameterTypes.length == 1 && parameterTypes[0].getName().equals(PAGEABLE)) {
            // Any method with Pageable as parameter is OK
            return BindingType.LAZY;
        }

        if (parameterTypes.length == 2 && parameterTypes[0].getName().equals(PAGEABLE)
                && (parameterTypes[1].getName().equals(FLOW_FILTER)
                        || parameterTypes[1].getName().equals(HILLA_FILTER))) {
            // Using CrudRepositoryService with Pageable and Filter is OK
            return BindingType.LAZY;
        }

        return BindingType.NONE;
    }

    /**
     * Checks if a Hilla Grid can be bound to the given service method.
     *
     * @param serviceMethod
     *            the service method to check
     * @return the binding type that can be used
     */
    public static BindingType canBindHillaGridToService(SpringBridge.ServiceMethodInfo serviceMethod) {
        return canBindFlowGridToService(serviceMethod);
    }

    public record ParametersAndImports(List<String> parameters, List<String> imports) {
    }

    /**
     * Returns the parameters to be used when calling the service method for binding
     * to a Flow Grid.
     * 
     * @param serviceMethod
     *            the service method to bind to
     * @return the parameters to be used when calling the service method
     */
    public static ParametersAndImports getParametersForGridBinding(SpringBridge.ServiceMethodInfo serviceMethod) {
        Method method = serviceMethod.serviceMethod();
        Class<?>[] parameterTypes = method.getParameterTypes();
        if (parameterTypes.length == 0) {
            return new ParametersAndImports(List.of(), List.of());
        }
        if (parameterTypes.length == 1 && parameterTypes[0].getName().equals(PAGEABLE)) {
            return new ParametersAndImports(List.of("VaadinSpringDataHelpers.toSpringPageRequest(query)"),
                    List.of("com.vaadin.flow.spring.data.VaadinSpringDataHelpers"));
        }
        if (parameterTypes.length == 2 && parameterTypes[0].getName().equals(PAGEABLE)
                && (parameterTypes[1].getName().equals(FLOW_FILTER)
                        || parameterTypes[1].getName().equals(HILLA_FILTER))) {
            return new ParametersAndImports(List.of("VaadinSpringDataHelpers.toSpringPageRequest(query)", "null"),
                    List.of("com.vaadin.flow.spring.data.VaadinSpringDataHelpers"));
        }
        throw new IllegalArgumentException("Unable to bind to the given service method");
    }

}
