package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.ide.IdeUtils;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaBatchRewriter;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaDataProviderHandler;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaRewriterCopyPasteHandler;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.copilot.javarewriter.SourceSyncChecker;
import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;
import com.vaadin.flow.server.VaadinServletContext;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import elemental.json.JsonType;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.javaparser.ast.expr.MethodCallExpr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles commands to rewrite Java source code.
 */
public class JavaRewriteHandler implements CopilotCommand {

    private final ObjectMapper objectMapper = new ObjectMapper();
    public static final String UNDO_LABEL = CopilotIDEPlugin.undoLabel("Java view update");
    private static final String CAN_BE_EDITED = "canBeEdited";
    private static final String IS_TRANSLATION = "isTranslation";
    private static final String COMPONENT_ID = "componentId";
    private final ProjectManager projectManager;
    private final SourceSyncChecker sourceSyncChecker;
    private final ComponentSourceFinder sourceFinder;
    private final VaadinServletContext vaadinServletContext;
    private static final String COMPONENT_PROPERTY = "component";
    private static final String PROPERTY_TO_CHECK_PROPERTY = "propertyToCheck";

    private static final String[] supportedEditableProperties = new String[] { "label", "helperText", "text", "title" };

    private interface Handler {
        void handle(JsonObject data, JsonObject respData) throws IOException;
    }

    private static class RewriteHandler {
        private final String what;
        private final Handler handler;

        public RewriteHandler(String what, Handler handler) {
            this.what = what;
            this.handler = handler;
        }

        public void handle(JsonObject data, JsonObject respData) throws IOException {
            this.handler.handle(data, respData);
        }

        public String getWhat() {
            return what;
        }
    }

    private final Map<String, RewriteHandler> handlers = new HashMap<>();

    /**
     * Creates the one and only handler.
     *
     * @param projectManager
     *            the project manager
     * @param sourceSyncChecker
     *            the source sync checker for detecting out of sync scenarios
     */
    public JavaRewriteHandler(ProjectManager projectManager, SourceSyncChecker sourceSyncChecker,
            VaadinServletContext vaadinServletContext) {
        this.projectManager = projectManager;
        this.sourceSyncChecker = sourceSyncChecker;
        this.sourceFinder = new ComponentSourceFinder(projectManager);
        this.vaadinServletContext = vaadinServletContext;

        handlers.put("set-component-property",
                new RewriteHandler("set component property", this::handleSetComponentProperty));
        handlers.put("add-call", new RewriteHandler("add call", this::handleAddCall));
        handlers.put("add-template", new RewriteHandler("add call", this::handleAddTemplate));
        handlers.put("delete-components", new RewriteHandler("delete components", this::handleDeleteComponents));
        handlers.put("duplicate-components",
                new RewriteHandler("duplicate components", this::handleDuplicateComponents));
        handlers.put("drag-and-drop", new RewriteHandler("drop component", this::handleDragAndDrop));
        handlers.put("set-alignment", new RewriteHandler("set alignment", this::handleAlignment));
        handlers.put("set-gap", new RewriteHandler("set gap", this::handleGap));
        handlers.put("wrap-with", new RewriteHandler("wrap with", this::handleWrapWith));
        handlers.put("set-styles", new RewriteHandler("set styles", this::handleSetStyles));
        handlers.put("set-padding", new RewriteHandler("set padding", this::handlePadding));
        handlers.put("copy", new RewriteHandler("copy", this::handleCopy));
        handlers.put("can-be-edited", new RewriteHandler("can be edited", this::handleCanBeEdited));
        handlers.put("set-sizing", new RewriteHandler("set sizing", this::handleSetSizing));
        handlers.put("connect-to-service", new RewriteHandler("connect to service", this::handleConnectToService));
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        RewriteHandler handler = handlers.get(command);
        if (handler == null) {
            return false;
        }
        String reqId = data.getString(KEY_REQ_ID);
        JsonObject respData = Json.createObject();
        respData.put(KEY_REQ_ID, reqId);

        try {
            handler.handle(data, respData);
            devToolsInterface.send(command + "-response", respData);
        } catch (ComponentInfoNotFoundException e) {
            if (sourceSyncChecker.maybeOutOfSync(e)) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData,
                        e.getComponentTypeAndSourceLocation().javaFile().map(File::getName).orElse("?")
                                + " may be out of sync. Please recompile and deploy the file",
                        e);
            } else {
                getLogger().debug("Failed to {} for input {}", handler.getWhat(), data.toJson(), e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, "Failed to " + handler.getWhat(),
                        e);
            }
        } catch (Exception e) {
            getLogger().debug("Failed to {} for input {}", handler.getWhat(), data.toJson(), e);
            ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, "Failed to " + handler.getWhat(), e);
        }

        return true;
    }

    private void handleAddCall(JsonObject data, JsonObject respData) throws IOException {
        String func = data.getString("func");
        String parameter = data.getString("parameter");
        Integer lineToShowInIde = data.hasKey("lineToShowInIde") ? (int) data.getNumber("lineToShowInIde") : null;
        var component = data.getObject(COMPONENT_PROPERTY);

        ComponentTypeAndSourceLocation typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        File javaFile = projectManager.getSourceFile(typeAndSourceLocation.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo info = rewriter.findComponentInfo(typeAndSourceLocation);
        if (!JavaRewriterUtil.hasSingleParameterMethod(info.type(), func)) {
            throw new IllegalArgumentException(
                    "Component " + info.type().getName() + " does not support the given method " + func);
        }
        rewriter.addCall(info, func, new JavaRewriter.Code(parameter));
        String result = rewriter.getResult();
        projectManager.writeFile(javaFile, UNDO_LABEL, result);

        if (lineToShowInIde != null) {
            int lineNumber = rewriter.getFirstModifiedRow() + lineToShowInIde;
            IdeUtils.openFile(javaFile, lineNumber);
            CompletableFuture.runAsync(() -> {
                try {
                    // Workaround for
                    // https://youtrack.jetbrains.com/issue/IDEA-342750
                    Thread.sleep(1000);
                    IdeUtils.openFile(javaFile, lineNumber);
                } catch (InterruptedException e) {
                    getLogger().error("Failed to show file in IDE", e);
                    Thread.currentThread().interrupt();
                }
            });
        }
    }

    private void handleDeleteComponents(JsonObject data, JsonObject respData) {
        JsonArray componentsJson = data.getArray("components");
        List<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        for (int i = 0; i < componentsJson.length(); i++) {
            List<ComponentTypeAndSourceLocation> found = findTypeAndSourceLocationIncludingChildren(
                    componentsJson.getObject(i));
            components.addAll(found);
        }

        JavaBatchRewriter batchRewriter = new JavaBatchRewriter(projectManager, components);

        batchRewriter.deleteAll();
        batchRewriter.writeResult();
    }

    private List<ComponentTypeAndSourceLocation> findTypeAndSourceLocationIncludingChildren(JsonObject object) {
        ArrayList<ComponentTypeAndSourceLocation> all = new ArrayList<>();
        ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(object, true);

        addRecursively(all, root);
        return all;
    }

    private void addRecursively(ArrayList<ComponentTypeAndSourceLocation> all, ComponentTypeAndSourceLocation root) {
        all.add(root);
        for (ComponentTypeAndSourceLocation child : root.children()) {
            if (child.javaFile().map(File::exists).orElse(false)) {
                addRecursively(all, child);
            } else {
                getLogger().debug(
                        "Excluding file {} because it does not exist. Assuming this is a component created internally by the parent component and not from the project source",
                        child.javaFile());
            }
        }
    }

    private void handleCopy(JsonObject data, JsonObject response) throws IOException {
        int componentId = (int) data.getNumber(COMPONENT_ID);

        int uiId = (int) data.getNumber("uiId");
        ComponentTypeAndSourceLocation copiedComponent = sourceFinder.findTypeAndSourceLocation(uiId, componentId,
                true);

        JavaRewriterCopyPasteHandler handler = new JavaRewriterCopyPasteHandler(projectManager);
        JavaComponent copiedJavaComponent = handler.getCopiedJavaComponent(copiedComponent);
        String s = objectMapper.writeValueAsString(copiedJavaComponent);
        response.put(COMPONENT_PROPERTY, s);
    }

    private void handleDuplicateComponents(JsonObject data, JsonObject respData) {
        JsonArray componentsJson = data.getArray("components");
        ArrayList<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        List<ComponentTypeAndSourceLocation> selectedComponents = new ArrayList<>();
        for (int i = 0; i < componentsJson.length(); i++) {
            ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(componentsJson.getObject(i),
                    true);
            selectedComponents.add(root);
            addRecursively(components, root);
        }

        JavaBatchRewriter batchRewriter = new JavaBatchRewriter(projectManager, components);
        selectedComponents.forEach(batchRewriter::duplicate);
        batchRewriter.writeResult();
    }

    private void handleWrapWith(JsonObject data, JsonObject respData) throws IOException {
        JsonArray componentsJson = data.getArray("components");
        JavaComponent wrapperComponent = JavaComponent.componentFromJson(data.getObject("wrapperComponent"));
        List<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        for (int i = 0; i < componentsJson.length(); i++) {
            components.add(sourceFinder.findTypeAndSourceLocation(componentsJson.getObject(i)));
        }
        File javaFile = projectManager.getSourceFile(components.get(0).getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));

        List<ComponentInfo> componentInfos = components.stream().map(rewriter::findComponentInfo).toList();

        rewriter.mergeAndReplace(componentInfos, wrapperComponent);
        String result = rewriter.getResult();
        projectManager.writeFile(javaFile, UNDO_LABEL, result);
    }

    private void handleSetComponentProperty(JsonObject data, JsonObject respData) throws IOException {
        String property = data.getString("property");
        String value = data.getString("value");
        var component = data.getObject("component");

        ComponentTypeAndSourceLocation typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        if (JavaDataProviderHandler.isDataProviderItemChange(typeAndSourceLocation)) {
            JavaDataProviderHandler dataProviderHandler = new JavaDataProviderHandler(projectManager,
                    typeAndSourceLocation);
            JavaDataProviderHandler.JavaDataProviderHandlerResult javaDataProviderHandlerResult = dataProviderHandler
                    .handleSetComponentProperty(property, value);
            projectManager.writeFile(javaDataProviderHandlerResult.file(), UNDO_LABEL,
                    javaDataProviderHandlerResult.result());
            return;
        }
        File javaFile = projectManager.getSourceFile(typeAndSourceLocation.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo info = rewriter.findComponentInfo(typeAndSourceLocation);
        String setter = JavaRewriterUtil.getSetterName(property, info.type(), true);

        JavaRewriter.ReplaceResult result = rewriter.replaceFunctionCall(info, setter, value);
        if (result.variableRenamedTo() != null) {
            respData.put("variableRenamedTo", result.variableRenamedTo());
        }
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleAddTemplate(JsonObject data, JsonObject respData) throws IOException {
        List<JavaComponent> template = JavaComponent.componentsFromJson(data.getArray("template"));
        JavaRewriter.Where where = JavaRewriter.Where.valueOf(data.getString("where").toUpperCase(Locale.ENGLISH));

        ComponentTypeAndSourceLocation refSource = sourceFinder.findTypeAndSourceLocation(data.getObject("refNode"));

        File javaFile = projectManager.getSourceFile(refSource.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo ref = rewriter.findComponentInfo(refSource);
        JavaRewriter.AddTemplateOptions options = new JavaRewriter.AddTemplateOptions(
                data.getBoolean("javaFieldsForLeafComponents"));
        if (where == JavaRewriter.Where.APPEND) {
            rewriter.addComponentUsingTemplate(ref, where, template, options);
        } else {
            if (!refSource.parent().javaFile().equals(refSource.javaFile())) {
                throw new IllegalArgumentException(
                        "Cannot insert before a component in one file (" + refSource.javaFile()
                                + ") when the parent is in another file (" + refSource.parent().javaFile() + ")");
            }
            rewriter.addComponentUsingTemplate(ref, where, template, options);
        }
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleDragAndDrop(JsonObject data, JsonObject respData) throws IOException {
        JavaRewriter.Where where = JavaRewriter.Where.valueOf(data.getString("where").toUpperCase(Locale.ENGLISH));

        ComponentTypeAndSourceLocation dragged = sourceFinder.findTypeAndSourceLocation(data.getObject("dragged"));
        ComponentTypeAndSourceLocation container = sourceFinder.findTypeAndSourceLocation(data.getObject("container"));

        ComponentTypeAndSourceLocation insertBefore = where == JavaRewriter.Where.BEFORE
                ? sourceFinder.findTypeAndSourceLocation(data.getObject("insertBefore"))
                : null;

        File javaFile = projectManager.getSourceFile(container.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));

        if (!dragged.javaFile().equals(container.javaFile())) {
            throw new IllegalArgumentException("Cannot move a component in one file (" + dragged.javaFile()
                    + ") to another file (" + container.javaFile() + ")");
        }

        ComponentInfo draggedRef = rewriter.findComponentInfo(dragged);
        ComponentInfo containerRef = rewriter.findComponentInfo(container);
        ComponentInfo insertBeforeRef = insertBefore == null ? null : rewriter.findComponentInfo(insertBefore);

        rewriter.moveComponent(draggedRef, containerRef, insertBeforeRef, where);
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleAlignment(JsonObject data, JsonObject respData) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));

        String alignItemsClassName = safeGet(data, "alignItemsClassName");
        String justifyContentClassName = safeGet(data, "justifyContentClassName");

        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);
        rewriter.setAlignment(componentInfo, alignItemsClassName, justifyContentClassName);
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleSetStyles(JsonObject data, JsonObject respData) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));
        JsonArray added = data.getArray("added");
        JsonArray removed = data.getArray("removed");
        Set<String> toRemove = new HashSet<>();
        for (int i = 0; i < removed.length(); i++) {
            toRemove.add(removed.getObject(i).getString("key"));
        }

        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);

        for (int i = 0; i < added.length(); i++) {
            JsonObject rule = added.getObject(i);
            String key = rule.getString("key");
            String value = rule.getString("value");
            rewriter.setStyle(componentInfo, key, value);
            toRemove.remove(key);
        }

        for (String key : toRemove) {
            rewriter.setStyle(componentInfo, key, null);
        }
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleConnectToService(JsonObject data, JsonObject respData) throws IOException {
        if (!SpringBridge.isSpringAvailable(vaadinServletContext)) {
            throw new IllegalStateException("Connecting to a service is only supported for Spring applications");
        }

        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));
        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);
        JsonObject service = data.getObject("service");
        String className = service.getString("className");
        String methodName = service.getString("methodName");

        List<SpringBridge.ServiceMethodInfo> flowUIServices = SpringBridge.getFlowUIServices(vaadinServletContext);
        SpringBridge.ServiceMethodInfo serviceMethod = flowUIServices.stream()
                .filter(serviceMethodInfo -> serviceMethodInfo.serviceClass().getName().equals(className)
                        && serviceMethodInfo.serviceMethod().getName().equals(methodName))
                .findFirst().orElseThrow(() -> new IllegalArgumentException("Service method not found"));

        JavaReflectionUtil.TypeInfo methodReturnType = JavaReflectionUtil.getReturnType(serviceMethod.serviceMethod(),
                serviceMethod.serviceClass());
        if (methodReturnType.typeParameters().size() != 1) {
            throw new IllegalArgumentException("Method return type " + methodReturnType + " is not a List of items");
        }
        JavaReflectionUtil.TypeInfo itemTypeInfo = methodReturnType.typeParameters().get(0);
        // Grid<Something> -> Grid<Product>
        Class<?> itemTypeClass;
        try {
            itemTypeClass = Class.forName(itemTypeInfo.typeName());
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException("Unable to find item class " + itemTypeInfo.typeName(), e);
        }

        List<String> propertiesInSourceOrder = JavaRewriterUtil.getPropertiesInSourceOrder(itemTypeClass,
                projectManager);
        propertiesInSourceOrder.remove("id"); // This is very common to have and you don't want to show it
        rewriter.setGridDataSource(componentInfo, serviceMethod, itemTypeClass, propertiesInSourceOrder);

        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleSetSizing(JsonObject data, JsonObject respData) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));
        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);
        JsonObject changesJson = data.getObject("changes");
        TypeReference<HashMap<String, String>> typeRef = new TypeReference<>() {
        };
        HashMap<String, String> changes = objectMapper.readValue(changesJson.toJson(), typeRef);

        rewriter.setSizing(componentInfo, changes);

        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleGap(JsonObject data, JsonObject respData) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));
        String lumoClassAll = safeGet(data, "all");
        String lumoClassRow = safeGet(data, "row");
        String lumoClassColumn = safeGet(data, "column");
        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);
        rewriter.setGap(componentInfo, lumoClassAll, lumoClassColumn, lumoClassRow);
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handlePadding(JsonObject data, JsonObject respData) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));
        String allPaddingLumoClass = safeGet(data, "all");
        String topPaddingLumoClass = safeGet(data, "top");
        String rightPaddingLumoClass = safeGet(data, "right");
        String bottomPaddingLumoClass = safeGet(data, "bottom");
        String leftPaddingLumoClass = safeGet(data, "left");
        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);
        rewriter.setPadding(componentInfo, allPaddingLumoClass, topPaddingLumoClass, rightPaddingLumoClass,
                bottomPaddingLumoClass, leftPaddingLumoClass);
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    // TODO remove this and use a proper way.
    private String safeGet(JsonObject data, String key) {
        return data.hasKey(key) && !data.get(key).getType().equals(JsonType.NULL) ? data.getString(key) : null;
    }

    private void handleCanBeEdited(JsonObject data, JsonObject response) throws IOException {
        var component = data.getObject(COMPONENT_PROPERTY);
        String propertyToCheck = data.getString(PROPERTY_TO_CHECK_PROPERTY);

        if (Arrays.stream(supportedEditableProperties)
                .noneMatch(property -> property.equalsIgnoreCase(propertyToCheck))) {
            response.put(CAN_BE_EDITED, false);
            response.put(IS_TRANSLATION, false);
            return;
        }
        var typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        var javaFile = projectManager.getSourceFile(typeAndSourceLocation.getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);
        var rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        var info = rewriter.findComponentInfo(typeAndSourceLocation);

        try {
            var result = rewriter.getPropertyValue(info, propertyToCheck);
            response.put(CAN_BE_EDITED, result instanceof String);
            if (result instanceof MethodCallExpr methodCallExpr) {
                response.put(IS_TRANSLATION, methodCallExpr.getNameAsString().equals("translate"));
            } else {
                response.put(IS_TRANSLATION, false);
            }
        } catch (Exception e) {
            getLogger().error("Failed to check if property can be edited", e);
            response.put(CAN_BE_EDITED, false);
            response.put(IS_TRANSLATION, false);
        }
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
