package com.vaadin.copilot;

import java.io.File;
import java.util.Optional;
import java.util.function.Consumer;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.server.VaadinServletContext;

import elemental.json.Json;
import elemental.json.JsonObject;

/**
 * Handles the downloading and setting up hotswap related tools for the project
 */
public class HotswapDownloadHandler implements CopilotCommand {
    private final ProjectManager projectManager;
    static final String PROGRESS_PREFIX = "Progress: ";

    public HotswapDownloadHandler(ProjectManager projectManager) {
        this.projectManager = projectManager;
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("set-up-vs-code-hotswap")) {
            JsonObject returnData = Json.createObject();
            returnData.put("reqId", data.getString("reqId"));
            try {
                Consumer<String> statusUpdater = (String msg) -> {
                    JsonObject json = Json.createObject();
                    if (msg.startsWith(PROGRESS_PREFIX)) {
                        json.put("progress", Double.parseDouble(msg.substring(PROGRESS_PREFIX.length())));
                    } else {
                        json.put("message", msg);
                    }
                    devToolsInterface.send("set-up-vs-code-hotswap-status", json);
                };
                JetbrainsRuntimeUtil jetbrainsRuntimeUtil = new JetbrainsRuntimeUtil(projectManager);
                Optional<File> jbrArchive = jetbrainsRuntimeUtil.downloadLatestJBR(statusUpdater);
                if (jbrArchive.isEmpty()) {
                    String error = """
                            "No JetBrains Runtime found for your architecture (%key%).
                            Visit https://github.com/JetBrains/JetBrainsRuntime/releases to find a suitable one.

                            Please report this in %tracker% if you think Copilot should be able to figure it out."
                            """.replace("%key%", jetbrainsRuntimeUtil.getDownloadKey()).replace("%tracker%",
                            Copilot.ISSUE_TRACKER);
                    ErrorHandler.setErrorMessage(returnData, error);
                    devToolsInterface.send(command + "-response", returnData);
                    return true;
                }
                File jdkFolder = jetbrainsRuntimeUtil.unpackJbr(jbrArchive.get(), statusUpdater);
                File hotswapAgentJarLocation = jetbrainsRuntimeUtil.getHotswapAgentLocation(jdkFolder);
                new HotswapAgentDownloader(projectManager).downloadHotswapAgent(hotswapAgentJarLocation, statusUpdater);

                statusUpdater.accept("Creating VS Code launch configuration");
                VsCodeUtil.ensureAutoHotDeploy(projectManager.getProjectRoot());
                VsCodeUtil.addHotswapAgentLaunchConfiguration(projectManager.getProjectRoot(),
                        jetbrainsRuntimeUtil.getJavaExecutable(jdkFolder),
                        (VaadinServletContext) projectManager.getVaadinSession().getService().getContext());
                devToolsInterface.send(command + "-response", returnData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData,
                        "Error downloading JetBrains Runtime", e);
            }
            return true;
        }
        return false;
    }
}
