package com.vaadin.copilot;

import static com.vaadin.copilot.Util.getFilenameFromClassName;
import static com.vaadin.copilot.Util.getGetterName;
import static com.vaadin.copilot.Util.getPackageName;
import static com.vaadin.copilot.Util.getSetterName;
import static com.vaadin.copilot.Util.getSimpleName;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CopyOnWriteArrayList;

import com.vaadin.flow.server.auth.AnonymousAllowed;

import elemental.json.Json;
import elemental.json.JsonObject;

import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.stmt.BlockStmt;

/**
 * UIServiceCreator is responsible for generating Java service and bean classes
 * dynamically. It supports different data storage types and provides
 * functionality for creating Java classes with fields, getters, and setters.
 *
 * <p>
 * This class utilizes JavaParser for AST modifications and dynamically writes
 * Java files to the project structure.
 */
public class UIServiceCreator {
    private final ProjectManager projectManager;

    public enum DataStorage {
        JPA, IN_MEMORY
    }

    /**
     * Represents information about a field in a Java bean.
     *
     * @param name
     *            the name of the property
     * @param javaType
     *            the Java type of the property
     */
    public record FieldInfo(String name, String javaType) {
        public static Object fromJson(JsonObject obj) {
            return new UIServiceCreator.FieldInfo(obj.getString("name"), obj.getString("javaType"));
        }

        public JsonObject toJson() {
            JsonObject obj = Json.createObject();
            obj.put("name", name());
            obj.put("javaType", javaType());
            return obj;
        }
    }

    /**
     * Represents information about a Java bean.
     *
     * @param fullyQualifiedBeanName
     *            the fully qualified name of the bean class
     * @param fields
     *            the fields of the bean
     */
    public record BeanInfo(String fullyQualifiedBeanName, FieldInfo[] fields) {
    }

    /**
     * Represents information about a service and its associated bean.
     *
     * @param beanInfo
     *            the information about the bean
     * @param dataStorage
     *            the type of data storage
     * @param browserCallable
     *            whether the service can be called from the browser
     * @param generateExampleData
     *            whether to generate example data
     */
    public record ServiceAndBeanInfo(BeanInfo beanInfo, DataStorage dataStorage, boolean browserCallable,
            boolean generateExampleData, String servicePackageName, String repositoryPackageName) {

        public String getBeanName() {
            return beanInfo().fullyQualifiedBeanName();
        }

        public String getRepositoryName() {
            String base;
            if (repositoryPackageName() != null) {
                base = repositoryPackageName() + "." + getSimpleName(getBeanName());
            } else {
                base = getBeanName();
            }
            return base + "Repository";
        }

        public String getServiceName() {
            String base;
            if (servicePackageName() != null) {
                base = servicePackageName() + "." + getSimpleName(getBeanName());
            } else {
                base = getBeanName();
            }
            return base + "Service";
        }
    }

    /**
     * Constructs a new UIServiceCreator with the given ProjectManager.
     *
     * @param projectManager
     *            the project manager to use for file operations
     */
    public UIServiceCreator(ProjectManager projectManager) {
        this.projectManager = projectManager;
    }

    /**
     * Creates a service and bean based on the provided ServiceAndBeanInfo.
     *
     * @param serviceAndBeanInfo
     *            the information about the service and bean to create
     * @param moduleInfo
     *            the module to create the file in
     * @throws IOException
     *             if an I/O error occurs
     */
    public void createServiceAndBean(ServiceAndBeanInfo serviceAndBeanInfo,
            JavaSourcePathDetector.ModuleInfo moduleInfo) throws IOException {
        createBean(serviceAndBeanInfo.beanInfo(), serviceAndBeanInfo.dataStorage, moduleInfo);
        createService(serviceAndBeanInfo, moduleInfo);
        if (serviceAndBeanInfo.dataStorage() == DataStorage.JPA) {
            createRepository(serviceAndBeanInfo, moduleInfo);
        }
    }

    /**
     * Creates a Java bean class based on the provided BeanInfo.
     *
     * @param beanInfo
     *            the information about the bean to create
     * @param moduleInfo
     *            the module to create the file in
     * @throws IOException
     *             if an I/O error occurs
     */
    void createBean(BeanInfo beanInfo, DataStorage dataStorage, JavaSourcePathDetector.ModuleInfo moduleInfo)
            throws IOException {
        String beanName = beanInfo.fullyQualifiedBeanName();
        String filename = getFilenameFromClassName(beanName);
        File javaFile = new File(moduleInfo.javaSourcePaths().get(0).toFile(), filename);
        ClassOrInterfaceDeclaration clazz = createClass(beanName);
        CompilationUnit compilationUnit = clazz.findCompilationUnit().orElseThrow();

        if (dataStorage == DataStorage.JPA) {
            // Add JPA entity annotations and imports
            compilationUnit.addImport("jakarta.persistence.Entity");
            clazz.addMarkerAnnotation("Entity");
        }

        // Add fields
        for (FieldInfo field : beanInfo.fields()) {
            FieldDeclaration fieldDeclaration = clazz.addField(getSimpleName(field.javaType()), field.name(),
                    Modifier.Keyword.PRIVATE);
            if (dataStorage == DataStorage.JPA && field.name().equals("id")) {
                compilationUnit.addImport("jakarta.persistence.Id");
                fieldDeclaration.addMarkerAnnotation("Id");
            }
        }

        // Generate getters and setters
        for (FieldInfo field : beanInfo.fields()) {
            String simpleJavaType = getSimpleName(field.javaType());
            compilationUnit.addImport(field.javaType());

            // Getter method
            MethodDeclaration getter = clazz.addMethod(getGetterName(field.name, field.javaType),
                    com.github.javaparser.ast.Modifier.Keyword.PUBLIC);
            getter.setType(simpleJavaType);
            getter.setBody(new BlockStmt().addStatement("return " + field.name() + ";"));

            // Setter method
            MethodDeclaration setter = clazz.addMethod(getSetterName(field.name), Modifier.Keyword.PUBLIC);
            setter.addParameter(simpleJavaType, field.name());
            setter.setBody(new BlockStmt().addStatement("this." + field.name() + " = " + field.name() + ";"));
        }
        projectManager.writeFile(javaFile, "New bean", compilationUnit.toString());
    }

    /**
     * Creates a service class based on the provided ServiceAndBeanInfo.
     *
     * @param serviceAndBeanInfo
     *            the information about the service and bean to create
     * @param moduleInfo
     *            the module to create the file in
     * @throws IOException
     *             if an I/O error occurs
     */
    private void createService(ServiceAndBeanInfo serviceAndBeanInfo, JavaSourcePathDetector.ModuleInfo moduleInfo)
            throws IOException {
        String beanName = serviceAndBeanInfo.getBeanName();
        String serviceName = serviceAndBeanInfo.getServiceName();
        String repositoryName = serviceAndBeanInfo.getRepositoryName();

        String filename = getFilenameFromClassName(serviceName);
        File javaFile = new File(moduleInfo.javaSourcePaths().get(0).toFile(), filename);
        ClassOrInterfaceDeclaration clazz = createClass(serviceName);
        CompilationUnit compilationUnit = clazz.findCompilationUnit().orElseThrow();

        String simpleBeanName = getSimpleName(beanName);
        String simpleRepositoryName = getSimpleName(repositoryName);

        compilationUnit.addImport("org.springframework.stereotype.Service");
        compilationUnit.addImport(beanName);
        compilationUnit.addImport("org.springframework.data.domain.Pageable");

        clazz.addMarkerAnnotation("Service");
        if (serviceAndBeanInfo.browserCallable()) {
            compilationUnit.addImport("com.vaadin.hilla.BrowserCallable");
            clazz.addMarkerAnnotation("BrowserCallable");
            compilationUnit.addImport(AnonymousAllowed.class);
            clazz.addMarkerAnnotation(AnonymousAllowed.class.getSimpleName());
        }
        if (serviceAndBeanInfo.dataStorage() == DataStorage.JPA) {
            compilationUnit.addImport(repositoryName);
            compilationUnit.addImport(List.class);

            // Add repository field
            clazz.addField(simpleRepositoryName, "repository", Modifier.Keyword.PRIVATE, Modifier.Keyword.FINAL);
            // Add constructor
            ConstructorDeclaration constructor = clazz.addConstructor(Modifier.Keyword.PUBLIC);
            constructor.addParameter(simpleRepositoryName, "repository");
            constructor.setBody(new BlockStmt().addStatement("this.repository = repository;"));

            // Add list method
            MethodDeclaration listMethod = clazz.addMethod("list", Modifier.Keyword.PUBLIC);
            listMethod.addParameter("Pageable", "pageable");
            listMethod.setType("List<" + simpleBeanName + ">");
            listMethod.setBody(new BlockStmt().addStatement("return repository.findAll(pageable).getContent();"));
        } else if (serviceAndBeanInfo.dataStorage() == DataStorage.IN_MEMORY) {
            // Add in-memory service implementation
            compilationUnit.addImport(CopyOnWriteArrayList.class);
            compilationUnit.addImport(List.class);
            compilationUnit.addImport(Objects.class);
            compilationUnit.addImport(Optional.class);

            // Add static data list
            clazz.addField("List<" + simpleBeanName + ">", "data", Modifier.Keyword.PRIVATE, Modifier.Keyword.STATIC,
                    Modifier.Keyword.FINAL).getVariables().get(0).setInitializer("new CopyOnWriteArrayList<>()");

            // Add list method
            MethodDeclaration listMethod = clazz.addMethod("list", Modifier.Keyword.PUBLIC);
            listMethod.addParameter("Pageable", "pageable");
            listMethod.setType("List<" + simpleBeanName + ">");
            listMethod.setBody(new BlockStmt().addStatement("int from = (int) pageable.getOffset();")
                    .addStatement(
                            "int to = (int) Math.min(pageable.getOffset() + pageable.getPageSize(), data.size());")
                    .addStatement("return data.subList(from, to);"));

            // Add save method
            MethodDeclaration saveMethod = clazz.addMethod("save", Modifier.Keyword.PUBLIC);
            saveMethod.addParameter(simpleBeanName, "value");
            saveMethod.setType(simpleBeanName);
            String saveMethodBody = """
                        {
                        Optional<%s> existingItem = data.stream().filter(item -> Objects.equals(item.getId(), value.getId())).findFirst();
                        existingItem.ifPresentOrElse(item -> {
                            int index = data.indexOf(item);
                            data.set(index, value);
                        }, () -> {
                            Long maxId = data.stream().map(%s::getId).max(Long::compareTo).orElse(0L);
                            value.setId(maxId + 1L);
                            data.add(value);
                        });
                        return value;
                        }
                    """
                    .formatted(simpleBeanName, simpleBeanName);

            saveMethod.setBody(StaticJavaParser.parseBlock(saveMethodBody));
            // Add delete method
            MethodDeclaration deleteMethod = clazz.addMethod("delete", Modifier.Keyword.PUBLIC);
            deleteMethod.addParameter("Long", "id");
            deleteMethod.setType("void");
            deleteMethod
                    .setBody(new BlockStmt().addStatement("data.removeIf(item -> Objects.equals(item.getId(), id));"));
        }

        projectManager.writeFile(javaFile, "New service", compilationUnit.toString());
    }

    private ClassOrInterfaceDeclaration createClass(String className) {
        CompilationUnit compilationUnit = createCompilationUnit(className);
        return compilationUnit.addClass(getSimpleName(className));
    }

    private ClassOrInterfaceDeclaration createInterface(String className) {
        CompilationUnit compilationUnit = createCompilationUnit(className);
        return compilationUnit.addInterface(getSimpleName(className));
    }

    private CompilationUnit createCompilationUnit(String className) {
        CompilationUnit compilationUnit = new CompilationUnit();

        // Set package name
        String packageName = getPackageName(className);
        if (!packageName.isEmpty()) {
            compilationUnit.setPackageDeclaration(packageName);
        }
        return compilationUnit;
    }

    /**
     * Creates a repository class based on the provided ServiceAndBeanInfo.
     *
     * @param serviceAndBeanInfo
     *            the information about the service and bean to create
     * @param moduleInfo
     *            the module to create the file in
     * @throws IOException
     *             if an I/O error occurs
     */
    private void createRepository(ServiceAndBeanInfo serviceAndBeanInfo, JavaSourcePathDetector.ModuleInfo moduleInfo)
            throws IOException {
        if (serviceAndBeanInfo.dataStorage != DataStorage.JPA) {
            return;
        }

        String beanName = serviceAndBeanInfo.getBeanName();
        String repositoryName = serviceAndBeanInfo.getRepositoryName();

        String filename = getFilenameFromClassName(repositoryName);
        File javaFile = new File(moduleInfo.javaSourcePaths().get(0).toFile(), filename);
        ClassOrInterfaceDeclaration clazz = createInterface(repositoryName);
        CompilationUnit compilationUnit = clazz.findCompilationUnit().orElseThrow();

        String simpleBeanName = getSimpleName(beanName);

        // Add imports
        compilationUnit.addImport("org.springframework.data.jpa.repository.JpaRepository");
        compilationUnit.addImport("org.springframework.data.jpa.repository.JpaSpecificationExecutor");
        compilationUnit.addImport(beanName);

        // Extend JpaRepository and JpaSpecificationExecutor
        clazz.addExtendedType("JpaRepository<" + simpleBeanName + ", Long>");
        clazz.addExtendedType("JpaSpecificationExecutor<" + simpleBeanName + ">");

        projectManager.writeFile(javaFile, "New repository", compilationUnit.toString());
    }

}