package com.vaadin.copilot;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ai.AICommandHandler;
import com.vaadin.copilot.analytics.AnalyticsClient;
import com.vaadin.copilot.analytics.AnalyticsInterceptor;
import com.vaadin.copilot.feedback.FeedbackHandler;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.ide.IdeHandler;
import com.vaadin.copilot.ide.IdePluginCommandHandler;
import com.vaadin.copilot.ide.OpenComponentInIDE;
import com.vaadin.copilot.javarewriter.SourceSyncChecker;
import com.vaadin.copilot.plugins.accessibilitychecker.AccessibilityCheckerMessageHandler;
import com.vaadin.copilot.plugins.devsetup.DevSetupHandler;
import com.vaadin.copilot.plugins.docs.DocsHandler;
import com.vaadin.copilot.plugins.i18n.I18nHandler;
import com.vaadin.copilot.plugins.info.InfoHandler;
import com.vaadin.copilot.plugins.testgeneration.GenerateTestsHandler;
import com.vaadin.copilot.plugins.themeeditor.ThemeEditorMessageHandler;
import com.vaadin.copilot.plugins.vaadinversionupdate.VaadinVersionUpdateHandler;
import com.vaadin.copilot.userinfo.UserInfo;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.flow.di.Lookup;
import com.vaadin.flow.internal.DevModeHandlerManager;
import com.vaadin.flow.server.VaadinContext;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import elemental.json.Json;
import elemental.json.JsonObject;

/**
 * The main code for Copilot for a given VaadinSession instance.
 *
 * <p>
 * One instance of this class is created for each VaadinSession which mean it is
 * ok to use and store VaadinSession specific data in this class and command
 * classes it uses, and in project manager etc.
 */
public class CopilotSession {

    private final VaadinSession vaadinSession;

    ProjectManager projectManager;

    private List<CopilotCommand> commands;

    private final CopilotIDEPlugin idePlugin;

    private SourceSyncChecker sourceSyncChecker;

    /**
     * Create a new CopilotSession for the given VaadinSession.
     *
     * @param vaadinSession
     *            the VaadinSession
     * @param devToolsInterface
     *            used to send messages back to the browser
     * @throws IOException
     *             if an error occurs
     */
    public CopilotSession(VaadinSession vaadinSession, DevToolsInterface devToolsInterface) throws IOException {
        this.vaadinSession = vaadinSession;
        VaadinServletContext context = Copilot.getContext(vaadinSession);

        ApplicationConfiguration applicationConfiguration = ApplicationConfiguration.get(context);
        CopilotIDEPlugin.setFolderInLaunchedModule(applicationConfiguration.getProjectFolder().toPath());
        CopilotIDEPlugin.setDevToolsInterface(devToolsInterface);
        if (ProjectFileManager.getInstance() == null
                || ProjectFileManager.getInstance().getApplicationConfiguration() != applicationConfiguration) {
            ProjectFileManager.initialize(applicationConfiguration);
        }
        projectManager = new ProjectManager(applicationConfiguration, vaadinSession);
        idePlugin = CopilotIDEPlugin.getInstance();

        Optional<DevModeHandlerManager> devModeHandlerManagerOptional = getDevModeHandlerManager(context);
        if (!applicationConfiguration.isProductionMode() && sourceSyncChecker == null
                && devModeHandlerManagerOptional.isPresent()) {
            sourceSyncChecker = new SourceSyncChecker(projectManager, devModeHandlerManagerOptional.get());
        }
        UserInfo userInfo = UserInfoServerClient.getUserInfoWithLocalProKey();

        setupCommands(applicationConfiguration, context, userInfo);
    }

    public void handleConnect(DevToolsInterface devToolsInterface) {
        devToolsInterface.send(Copilot.PREFIX + "init", null);
        for (CopilotCommand copilotCommand : commands) {
            copilotCommand.handleConnect(devToolsInterface);
        }
    }

    /**
     * Handle a message from the client.
     *
     * @param command
     *            the command
     * @param data
     *            the data, specific to the command
     * @param devToolsInterface
     *            used to send messages back to the browser
     */
    public void handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command == null) {
            throw new NullPointerException("command is null");
        }

        boolean canBeParallel = false;
        canBeParallel = commands.stream().anyMatch(c -> c.canBeParallelCommand(command));
        if (canBeParallel) {
            handleMessageAsync(command, data, devToolsInterface);
        } else {
            handleMessageSync(command, data, devToolsInterface);
        }
    }

    private void handleMessageAsync(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        CompletableFuture<Boolean>[] futures = commands.stream()
                .map(copilotCommand -> CompletableFuture
                        .runAsync(() -> copilotCommand.handleMessage(command, data, devToolsInterface)))
                .toArray(CompletableFuture[]::new);
        CompletableFuture.allOf(futures).thenRunAsync(() -> {
            boolean handled = Stream.of(futures).anyMatch(CompletableFuture::join);
            if (!handled) {
                JsonObject respData = Json.createObject();
                if (data.hasKey(CopilotCommand.KEY_REQ_ID)) {
                    respData.put(CopilotCommand.KEY_REQ_ID, data.getString(CopilotCommand.KEY_REQ_ID));
                }
                data.put("error", "Unknown command " + command);
                devToolsInterface.send("unknown-command", data);
            }
        });
    }

    private void handleMessageSync(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        for (CopilotCommand copilotCommand : commands) {
            try {
                if (copilotCommand.handleMessage(command, data, devToolsInterface)) {
                    return;
                }
            } catch (Exception e) {
                // Something that should maybe have been caught by the handler but was not
                ErrorHandler.sendErrorResponse(devToolsInterface, command, data, "Error handling command", e);
                return;
            }
        }
        JsonObject respData = Json.createObject();
        if (data.hasKey(CopilotCommand.KEY_REQ_ID)) {
            respData.put(CopilotCommand.KEY_REQ_ID, data.getString(CopilotCommand.KEY_REQ_ID));
        }
        data.put("error", "Unknown command " + command);
        devToolsInterface.send("unknown-command", data);
    }

    private void setupCommands(ApplicationConfiguration applicationConfiguration, VaadinServletContext context,
            UserInfo userInfo) {
        commands = List.of(
                // This must be first as it is more of an interceptor than a
                // handler
                new AnalyticsInterceptor(), new IdeHandler(idePlugin),
                new ErrorHandler(AnalyticsClient.getInstance().isEnabled() && userInfo != null
                        && userInfo.canDataLeaveLocalhost()),
                new OpenComponentInIDE(projectManager), new ProjectFileHandler(projectManager),
                new AICommandHandler(projectManager), new UserInfoHandler(context), new I18nHandler(projectManager),
                new IdePluginCommandHandler(projectManager), new ApplicationInitializer(projectManager, context),
                new MachineConfigurationHandler(), new ThemeEditorMessageHandler(projectManager),
                new RouteHandler(projectManager, context), new UiServiceHandler(projectManager, context),
                new AccessibilityCheckerMessageHandler(projectManager), new InfoHandler(applicationConfiguration),
                new FeedbackHandler(), new JavaRewriteHandler(projectManager, sourceSyncChecker, context),
                new PreviewHandler(), new GenerateTestsHandler(projectManager), new JavaParserHandler(),
                new DocsHandler(), new DevSetupHandler(idePlugin), new HotswapDownloadHandler(projectManager),
                new ProjectInfoHandler(context),
                new VaadinVersionUpdateHandler(projectManager, applicationConfiguration));
    }

    private Optional<DevModeHandlerManager> getDevModeHandlerManager(VaadinContext context) {
        return Optional.ofNullable(context).map(ctx -> ctx.getAttribute(Lookup.class))
                .map(lu -> lu.lookup(DevModeHandlerManager.class));
    }
}
