package com.vaadin.copilot;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.MalformedInputException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.vaadin.copilot.analytics.AnalyticsClient;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.UsageStatistics;
import com.vaadin.flow.server.frontend.FrontendUtils;
import com.vaadin.flow.server.frontend.ThemeUtils;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProjectFileManager {
    private static ProjectFileManager instance = null;

    private final ApplicationConfiguration applicationConfiguration;
    private JavaSourcePathDetector.ProjectPaths projectPaths0 = null;

    public static synchronized ProjectFileManager initialize(ApplicationConfiguration applicationConfiguration) {
        if (instance == null || instance.applicationConfiguration != applicationConfiguration) {
            instance = new ProjectFileManager(applicationConfiguration);
        }
        return instance;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ProjectFileManager.class);
    }

    public static ProjectFileManager getInstance() {
        return instance;
    }

    ProjectFileManager(ApplicationConfiguration applicationConfiguration) {
        this.applicationConfiguration = applicationConfiguration;
    }

    public String readFile(String filename) throws IOException {
        return readFile(new File(filename));
    }

    public String readFile(Path filename) throws IOException {
        return readFile(filename.toFile());
    }

    public String readFile(File file) throws IOException {
        if (file == null) {
            throw new IllegalArgumentException("File cannot be null");
        }
        if (!file.exists()) {
            throw new FileNotFoundException(file.getAbsolutePath());
        }
        if (isFileInsideProject(file)) {
            try {
                return Files.readString(file.toPath(), StandardCharsets.UTF_8);
            } catch (MalformedInputException e) {
                return Files.readString(file.toPath(), StandardCharsets.ISO_8859_1);
            }
        } else {
            throw createFileIsNotInProjectException(file);
        }
    }

    private IllegalArgumentException createFileIsNotInProjectException(File file) {
        return new IllegalArgumentException(
                "File " + file.getPath() + " is not inside the project " + getProjectPaths());

    }

    public List<String> readLines(File file) throws IOException {
        if (!isFileInsideProject(file)) {
            throw createFileIsNotInProjectException(file);
        }
        return FileUtils.readLines(file, StandardCharsets.UTF_8);
    }

    boolean isFileInsideProject(File file) {
        return findModule(file).isPresent();
    }

    public static boolean isFileInside(File toCheck, File container) {
        if (!container.exists()) {
            return false;
        }
        if (toCheck.exists()) {
            Path pathToCheck = toCheck.toPath();
            try {
                return pathToCheck.toRealPath().startsWith(container.toPath().toRealPath().toString());
            } catch (IOException e) {
                getLogger().error("Unable to check if " + toCheck + " is inside " + container, e);
                return false;
            }
        } else {
            // New file
            return isFileInside(toCheck.getParentFile(), container);
        }
    }

    /**
     * Writes the given content to the given file inside the project.
     *
     * <p>
     * If the filename is absolute, it is used as is. Otherwise, it is resolved
     * relative to the project root.
     *
     * <p>
     * If the file is outside the project, an exception is thrown
     *
     * @param filename
     *            the filename to write to, absolute or relative to the project root
     * @param undoLabel
     *            the undo label for the change
     * @param content
     *            the content to write
     * @throws IOException
     *             if the file cannot be written
     */
    public void writeFile(String filename, String undoLabel, String content) throws IOException {
        writeFile(getAbsolutePath(filename), undoLabel, content);
    }

    /**
     * Writes the given content to the given file inside the project.
     *
     * <p>
     * If the filename is absolute, it is used as is. Otherwise, it is resolved
     * relative to the project root.
     *
     * <p>
     * If the file is outside the project, an exception is thrown
     *
     * @param file
     *            the filename to write to, absolute or relative to the project root
     * @param undoLabel
     *            the undo label for the change
     * @param content
     *            the content to write
     * @throws IOException
     *             if the file cannot be written
     */
    public void writeFile(File file, String undoLabel, String content) throws IOException {
        writeFile(file, undoLabel, content, false);
    }

    public void writeFile(Path file, String undoLabel, String content) throws IOException {
        writeFile(file.toFile(), undoLabel, content, false);
    }

    public File writeFileBase64(String filename, String undoLabel, String base64Content, boolean renameIfExists)
            throws IOException {
        var target = getAbsolutePath(filename);
        while (renameIfExists && target.exists()) {
            target = new File(target.getParentFile(), Util.increaseTrailingNumber(target.getName()));
        }
        writeFile(target, undoLabel, base64Content, true);
        return target;
    }

    private void writeFile(File file, String undoLabel, String content, boolean base64Encoded) throws IOException {
        if (!isFileInsideProject(file)) {
            throw createFileIsNotInProjectException(file);
        }

        File folder = file.getParentFile();
        if (!folder.exists() && !folder.mkdirs()) {
            throw new IOException("Unable to create folder " + folder.getAbsolutePath());
        }
        AnalyticsClient.getInstance().track("write-file",
                Map.of("binary", String.valueOf(base64Encoded), "type", FilenameUtils.getExtension(file.getName())));
        UsageStatistics.markAsUsed("copilot/write-file", CopilotVersion.getVersion());
        CopilotIDEPlugin idePlugin = CopilotIDEPlugin.getInstance();
        if (idePlugin.isActive()) {
            if (base64Encoded && idePlugin.supports(CopilotIDEPlugin.Commands.WRITE_BASE64)) {
                idePlugin.writeBase64File(file, undoLabel, content);
                return;
            } else if (!base64Encoded && idePlugin.supports(CopilotIDEPlugin.Commands.WRITE)) {
                idePlugin.writeFile(file, undoLabel, content);
                return;
            }
        }
        writeFileIfNeeded(file, content, base64Encoded);
    }

    private void writeFileIfNeeded(File file, String content, boolean base64Encoded) throws IOException {
        if (file.exists()) {
            String currentContent;
            if (base64Encoded) {
                byte[] data = FileUtils.readFileToByteArray(file);
                currentContent = base64Encode(data);
            } else {
                currentContent = FileUtils.readFileToString(file, StandardCharsets.UTF_8);
            }
            if (content.equals(currentContent)) {
                return;
            }
        }

        byte[] data;
        if (base64Encoded) {
            data = base64Decode(content);
        } else {
            data = content.getBytes(StandardCharsets.UTF_8);
        }
        if (file.exists()) {
            try (FileChannel channel = FileChannel.open(file.toPath(), StandardOpenOption.WRITE)) {
                int written = channel.write(ByteBuffer.wrap(data));
                channel.truncate(written);
            }
        } else {
            Files.write(file.toPath(), data);
        }
        // If the file was a resource, we can directly copy it to target
        // to make it available
        copyResourceToTarget(file);
    }

    private void copyResourceToTarget(File resourceFile) throws IOException {
        Optional<Path> maybeResourceFolder = findResourceFolder(resourceFile);
        if (maybeResourceFolder.isEmpty()) {
            // Not a resource
            return;
        }
        File resourceFolder = maybeResourceFolder.get().toFile();
        Optional<JavaSourcePathDetector.ModuleInfo> module = findModule(resourceFolder);
        if (module.isEmpty()) {
            getLogger().error("Unable to determine module for resource folder {}", resourceFolder);
            return;
        }

        String relativeResourceName = getRelativeName(resourceFile, resourceFolder);
        File target = new File(module.get().classesFolder().toFile(), relativeResourceName);
        FileUtils.copyFile(resourceFile, target);
    }

    private byte[] base64Decode(String content) {
        return Base64.getDecoder().decode(content);
    }

    private String base64Encode(byte[] data) {
        return Base64.getEncoder().encodeToString(data);
    }

    public String makeAbsolute(String projectRelativeFilename) throws IOException {
        Path projectRootPath = getProjectRoot().toPath().toRealPath();
        Path resolved = projectRootPath.resolve(projectRelativeFilename).toRealPath();
        if (!resolved.startsWith(projectRootPath)) {
            throw new IllegalArgumentException(
                    "File " + projectRelativeFilename + " is not inside the project at " + projectRootPath);
        }
        return resolved.toString();
    }

    public String makeRelative(String filename) throws IOException {
        Path projectRootPath = getProjectRoot().toPath().toRealPath();
        Path absolutePath = new File(filename).toPath().toRealPath();
        // If file is not inside project root, throw
        if (!absolutePath.startsWith(projectRootPath)) {
            throw new IllegalArgumentException("File " + filename + " is not inside the project at " + projectRootPath);
        }
        return projectRootPath.relativize(absolutePath).toString();
    }

    /**
     * Returns the name of the file, relative to the project root.
     *
     * @param projectFile
     *            the file
     * @return the relative name of the file
     */
    public String getProjectRelativeName(File projectFile) {
        return getRelativeName(projectFile, getProjectRoot());
    }

    /**
     * Returns the name of the file, relative to the given folder.
     *
     * @param projectFile
     *            the file
     * @param folder
     *            the folder to make relative to
     * @return the relative name of the file
     */
    public static String getRelativeName(File projectFile, File folder) {
        return folder.toPath().relativize(projectFile.toPath()).toString();
    }

    /**
     * Returns the Java file for the given class.
     *
     * @param cls
     *            the class
     * @return the file for the class
     */
    public File getFileForClass(Class<?> cls) {
        return getFileForClass(cls.getName());
    }

    /**
     * Returns the Java file for the given class.
     *
     * @param cls
     *            the class
     * @return the file for the class
     */
    public File getFileForClass(String cls) {
        if (cls.contains("$")) {
            cls = cls.substring(0, cls.indexOf("$"));
        }
        cls = cls.replace(".", File.separator);

        for (Path path : getSourceFolders()) {
            for (String extension : new String[] { "java", "kt" }) {
                String filename = cls + "." + extension;
                File file = new File(path.toFile(), filename);
                if (file.exists()) {
                    return file;
                }
            }
        }

        // This is to be compatible with existing code which assumes a non-null return
        // value even if the file does not exist
        return new File(getSourceFolders().get(0).toFile(), cls + ".java");
    }

    /**
     * Returns the source folders for the project.
     *
     * @return the source folders
     */
    public List<Path> getSourceFolders() {
        return getProjectPaths().allSourcePaths();
    }

    /**
     * Gets the path information for the project.
     * <p>
     * Synchronized as during startup there can be multiple threads trying to access
     * this information, e.g. the hotswap listener which will run many times in
     * parallel for many classes.
     *
     * @return the project paths
     */
    synchronized JavaSourcePathDetector.ProjectPaths getProjectPaths() {
        if (projectPaths0 == null) {
            projectPaths0 = JavaSourcePathDetector.detectProjectPaths(applicationConfiguration);
            getLogger().debug("Project folders detected: {}", projectPaths0);
        }
        return projectPaths0;
    }

    /**
     * Returns the resource folders for the project.
     *
     * @return the resource folders
     */
    public List<Path> getResourceFolders() {
        return getProjectPaths().allResourcePaths();
    }

    /**
     * Returns the Java file for the given component location.
     *
     * @param location
     *            the component location
     * @return the file for the class where the component is used
     */
    public File getSourceFile(ComponentTracker.Location location) {
        return getFileForClass(location.className());
    }

    /**
     * Returns the project base folder.
     * <p>
     * The project base folder is the common folder containing all module roots.
     *
     * @return the project root folder
     */
    public File getProjectRoot() {
        return getProjectPaths().basedir().toFile();
    }

    /**
     * Returns the frontend folder.
     *
     * @return the frontend folder
     */
    public File getFrontendFolder() {
        return FrontendUtils.getProjectFrontendDir(applicationConfiguration);
    }

    /**
     * Returns the java resource folder.
     *
     * @return the java resource folder.
     */
    public File getJavaResourceFolder() {
        return applicationConfiguration.getJavaResourceFolder();
    }

    /**
     * Gets current theme name
     *
     * @return optional theme name
     */
    public Optional<String> getThemeName() {
        return ThemeUtils.getThemeName(applicationConfiguration.getContext());
    }

    /**
     * Gets current theme folder
     *
     * @return optional theme folder
     */
    public Optional<File> getThemeFolder() {
        return getThemeName().map(t -> ThemeUtils.getThemeFolder(getFrontendFolder(), t));
    }

    private File getAbsolutePath(String filename) {
        File file = new File(filename);
        if (!file.isAbsolute()) {
            file = new File(getProjectRoot(), filename);
        }
        return file;
    }

    /**
     * Makes a string safe to use as a file name
     *
     * @param name
     *            the string to process
     * @return the sanitized string
     */
    public String sanitizeFilename(String name) {
        return name.replaceAll("[^a-zA-Z0-9-_@]", "_");
    }

    /**
     * Finds the folder where Hilla views should be created for the file system
     * router to pick them up.
     *
     * @return the folder where Hilla views should be created
     */
    public File getHillaViewsFolder() {
        return new File(getFrontendFolder(), "views");
    }

    /**
     * Finds the source folder where the given class is located.
     *
     * @param cls
     *            the class to find the source folder for
     * @return the source folder or an empty optional if the source folder could not
     *         be found
     * @throws IOException
     *             if something goes wrong
     */
    Optional<Path> findSourceFolder(Class<?> cls) {
        File sourceFile = getFileForClass(cls);
        if (!sourceFile.exists()) {
            return Optional.empty();
        }

        return findSourceFolder(sourceFile);
    }

    /**
     * Finds the given resource by name, in any resource folder.
     *
     * @param resource
     *            the name of the resource to look for
     */
    public Optional<Path> findResource(String resource) {
        for (Path resourceFolder : getResourceFolders()) {
            Path resourceFile = resourceFolder.resolve(resource);
            if (resourceFile.toFile().exists()) {
                return Optional.of(resourceFile);
            }
        }
        return Optional.empty();
    }

    /**
     * Finds the resource folder where the given resource is located.
     *
     * @param resourceFile
     *            the resource to find the resource folder for
     */
    public Optional<Path> findResourceFolder(File resourceFile) {
        for (Path resourceFolder : getResourceFolders()) {
            if (isFileInside(resourceFile, resourceFolder.toFile())) {
                return Optional.of(resourceFolder);
            }
        }
        return Optional.empty();
    }

    /**
     * Finds the module where the given resource is located.
     *
     * @param file
     *            the file to look up
     */
    public Optional<JavaSourcePathDetector.ModuleInfo> findModule(File file) {
        if (file == null) {
            return Optional.empty();
        }
        for (JavaSourcePathDetector.ModuleInfo moduleInfo : getProjectPaths().modules()) {
            if (isFileInside(file, moduleInfo.rootPath().toFile())) {
                return Optional.of(moduleInfo);
            }
        }
        return Optional.empty();
    }

    /**
     * Finds the source folder where the given file is located.
     *
     * @param sourceFile
     *            the source file to find the source folder for
     */
    public Optional<Path> findSourceFolder(File sourceFile) {
        for (Path sourceFolder : getSourceFolders()) {
            if (isFileInside(sourceFile, sourceFolder.toFile())) {
                return Optional.of(sourceFolder);
            }
        }
        return Optional.empty();
    }

    /**
     * Gets the java package for the given source file
     *
     * @param sourceFile
     *            the source file
     * @return the java package for the given source file, or null if the source
     *         file is not inside the project
     * @throws IOException
     *             if something goes wrong
     */
    public String getJavaPackage(File sourceFile) throws IOException {
        Optional<Path> sourceFolder = findSourceFolder(sourceFile);
        if (sourceFolder.isEmpty()) {
            return null;
        }
        return getRelativeName(sourceFile.getParentFile(), sourceFolder.get().toFile()).replace(File.separator, ".");
    }

    /**
     * Gets the modules in the project.
     *
     * @return a list of modules in the project
     */
    public List<JavaSourcePathDetector.ModuleInfo> getModules() {
        return getProjectPaths().modules();
    }

    public ApplicationConfiguration getApplicationConfiguration() {
        return applicationConfiguration;
    }
}
