package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;

import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.flow.shared.util.SharedUtil;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles creation of views for new routes.
 */
public class RouteCreator {

    private final ProjectManager projectManager;
    private final NewRouteTemplateHandler newRouteTemplateHandler = new NewRouteTemplateHandler();

    public RouteCreator(ProjectManager projectManager) {
        this.projectManager = projectManager;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(RouteCreator.class);
    }

    /**
     * Creates a Flow view for the given route using the given access requirement.
     *
     * @param route
     *            the path to create a view for
     * @param accessRequirement
     *            the access requirement for the view
     * @param layoutClass
     *            the layout class to use, or {@code null} to not use a layout
     * @param templateFileName
     *            the file name to select the template of view
     * @throws IOException
     *             if the view file could not be created
     */
    public void createFlowView(String route, AccessRequirement accessRequirement, Class<?> layoutClass,
            String templateFileName) throws IOException, RouteDuplicatedException {
        getLogger().debug("Creating Flow view for route {}", route);

        throwIfInvalidRoute(route);

        String lastRouteSegment = getLastRouteSegment(route);
        String className = getFlowViewName(route);
        if (!projectManager.sanitizeFilename(className).equals(className)) {
            throw new IllegalArgumentException("Invalid filename " + className);
        }

        File viewFolder = projectManager.getFlowNewViewFolder();
        File viewFile = new File(viewFolder, className + ".java");
        String viewsPackage = projectManager.getJavaPackage(viewFile);

        if (existsRoute(route)) {
            throw new RouteDuplicatedException(route);
        }
        if (viewFile.exists()) {
            throw new RouteDuplicatedException(route);
        }
        viewFile.getParentFile().mkdirs();

        String title = Util.titleCase(SharedUtil.dashSeparatedToCamelCase(lastRouteSegment));

        NewRouteTemplateHandler.FlowTemplateRequest flowTemplateRequest = new NewRouteTemplateHandler.FlowTemplateRequest(
                viewsPackage, route, title, className, accessRequirement, layoutClass, templateFileName);
        String content = newRouteTemplateHandler.getFlowTemplate(flowTemplateRequest);
        projectManager.writeFile(viewFile, CopilotIDEPlugin.undoLabel("Add route"), content);
    }

    private void throwIfInvalidRoute(String route) {
        if (route.isEmpty()) {
            return;
        }
        if (!route.matches("^[a-zA-Z0-9-/]*$")) {
            throw new IllegalArgumentException(
                    "Routes can only contain letters, numbers, dashes and separators (/): " + route);
        }
        if (!route.matches(".*[a-zA-Z0-9]+.*")) {
            throw new IllegalArgumentException("Route must contain at least one letter or number: " + route);
        }
        if (route.contains("//")) {
            throw new IllegalArgumentException("Route must not contain consecutive slashes: " + route);
        }
    }

    private static String getLastRouteSegment(String route) {
        if (route.contains("/")) {
            return route.substring(route.lastIndexOf('/') + 1);
        }
        return route;
    }

    static String getFlowViewName(String route) {
        String[] parts = route.split("/");
        String filename = parts[parts.length - 1];

        String identifier;
        if (filename.isEmpty() || filename.endsWith("/")) {
            identifier = "Main";
        } else {
            identifier = SharedUtil.capitalize(JavaRewriterUtil.getJavaIdentifier(filename, 100));
        }

        return identifier;
    }

    /**
     * Creates a Hilla view for the given route using the given access requirement.
     *
     * @param route
     *            the path to create a view for
     * @param accessRequirement
     *            the access requirement for the view
     * @param templateFileName
     *            the selected template for the view
     */
    public void createHillaView(String route, AccessRequirement accessRequirement, String templateFileName)
            throws IOException, RouteDuplicatedException {
        getLogger().debug("Creating Hilla view for route {}", route);

        throwIfInvalidRoute(route);
        // Assumes FS router will set up routing
        String filenameWithPath = route;
        if (filenameWithPath.isEmpty() || filenameWithPath.endsWith("/")) {
            filenameWithPath += "@index";
        }

        if (!projectManager.sanitizeFilename(filenameWithPath).equals(filenameWithPath.replace('/', '_'))) {
            throw new IllegalArgumentException("Invalid filename " + filenameWithPath);
        }
        File viewFile = projectManager.getHillaViewsFolder();

        String[] parts = filenameWithPath.split("/");
        for (String part : parts) {
            viewFile = new File(viewFile, part);
        }
        if (existsRoute(route)) {
            throw new RouteDuplicatedException(route);
        }
        viewFile = addExtension(viewFile, ".tsx");
        if (viewFile.exists()) {
            throw new RouteDuplicatedException(route);
        }
        viewFile.getParentFile().mkdirs();
        String viewName = viewFile.getName().replace(".tsx", "");
        if (viewName.startsWith("@")) {
            viewName = "";
        }
        String title = Util.titleCase(SharedUtil.dashSeparatedToCamelCase(viewName));
        NewRouteTemplateHandler.HillaTemplateRequest hillaTemplateRequest = new NewRouteTemplateHandler.HillaTemplateRequest(
                parts[parts.length - 1], title, accessRequirement, templateFileName);

        String content = newRouteTemplateHandler.getHillaTemplate(hillaTemplateRequest);
        projectManager.writeFile(viewFile, CopilotIDEPlugin.undoLabel("Add route"), content);
    }

    private static File addExtension(File viewFile, String ext) {
        return new File(viewFile.getParentFile(), viewFile.getName() + ext);
    }

    private boolean existsRoute(String route) {
        try {
            RouteHandler.getServerRoutes(projectManager.getVaadinSession()).stream()
                    .filter(routeData -> routeData.getTemplate().equals(route)).findAny().ifPresent(routeData -> {
                        throw new CopilotException("Route already exists: " + route);
                    });
        } catch (RuntimeException e) {
            return true;
        }
        return false;
    }
}
