package com.vaadin.copilot;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.auth.AnnotatedViewAccessChecker;
import com.vaadin.flow.server.auth.NavigationAccessControl;
import com.vaadin.hilla.EndpointRegistry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.UnsatisfiedDependencyException;
import org.springframework.boot.SpringBootVersion;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.cglib.proxy.Proxy;
import org.springframework.core.SpringVersion;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

/**
 * Provides Spring related helpers for copilot. Depends on Spring classes and
 * cannot be directly imported
 */
public class SpringIntegration {

    /**
     * Returns the value of the given property from the Spring environment of the
     * given context. See {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @param property
     *            the property name
     * @return the property value or null if not found
     */
    public static String getPropertyValue(VaadinServletContext context, String property) {
        Environment env = getWebApplicationContext(context).getEnvironment();
        return env.getProperty(property);
    }

    public static WebApplicationContext getWebApplicationContext(VaadinServletContext context) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            throw new IllegalStateException("No Spring web application context available");
        }
        return webAppContext;
    }

    /**
     * Returns the Spring Boot application class of the given context. See
     * {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @return the Spring Boot application class or null if not found
     */
    public static Class<?> getApplicationClass(VaadinServletContext context) {
        Map<String, Object> beans = getWebApplicationContext(context)
                .getBeansWithAnnotation(SpringBootApplication.class);
        Class<?> appClass = beans.values().iterator().next().getClass();
        if (Proxy.isProxyClass(appClass)) {
            appClass = appClass.getSuperclass();
        }
        while (isCglibProxy(appClass)) {
            appClass = appClass.getSuperclass();
        }
        return appClass;
    }

    private static boolean isCglibProxy(Class<?> appClass) {
        return appClass.getName().contains("$$SpringCGLIB$$");
    }

    /**
     * Returns whether Flow view security is enabled in the given context.
     *
     * @param context
     *            the Vaadin servlet context
     * @return true if Flow view security is enabled, false otherwise
     */
    public static Boolean isViewSecurityEnabled(VaadinServletContext context) {
        WebApplicationContext webApplicationContext = getWebApplicationContext(context);
        String[] naviAccessControl = webApplicationContext.getBeanNamesForType(NavigationAccessControl.class);
        if (naviAccessControl.length != 1) {
            return false;
        }
        NavigationAccessControl accessControl = (NavigationAccessControl) webApplicationContext
                .getBean(naviAccessControl[0]);
        return accessControl.hasAccessChecker(AnnotatedViewAccessChecker.class);
    }

    public static List<SpringBridge.ServiceMethodInfo> getEndpoints(VaadinServletContext context) {
        EndpointRegistry endpointRegistry = getBean(EndpointRegistry.class, context);
        Map<String, EndpointRegistry.VaadinEndpointData> vaadinEndpoints = endpointRegistry.getEndpoints();

        return getEndpointInfos(vaadinEndpoints);
    }

    public static List<SpringBridge.ServiceMethodInfo> getFlowUIServices(VaadinServletContext context) {
        try {
            Collection<Object> springServices = getWebApplicationContext(context).getBeansWithAnnotation(Service.class)
                    .values();
            Stream<Class<?>> serviceClassCandidates = springServices.stream()
                    .map(inst -> Proxy.isProxyClass(inst.getClass()) || isCglibProxy(inst.getClass())
                            ? inst.getClass().getSuperclass()
                            : inst.getClass());
            serviceClassCandidates = serviceClassCandidates
                    .filter(cls -> !cls.getPackage().getName().startsWith("com.vaadin"));

            Stream<SpringBridge.ServiceMethodInfo> serviceMethods = serviceClassCandidates
                    .flatMap(cls -> Arrays.stream(cls.getMethods()).filter(m -> !isObjectMethod(m))
                            .map(method -> new SpringBridge.ServiceMethodInfo(cls, method)));
            return serviceMethods.toList();
        } catch (UnsatisfiedDependencyException e) {
            // This happens if there is a problem in creating one service, e.g.
            // Error creating bean with name 'fooService' defined in ...:
            // Unsatisfied dependency expressed through constructor parameter 0: No
            // qualifying bean of type 'test.vaadin.copilot.hilla.data.FooRepository'
            // available:
            // expected at least 1 bean which qualifies as autowire candidate. Dependency
            // annotations: {}
            getLogger().trace("Unable to get flow UI services because of a Spring dependency issue. Ignoring it.", e);
            return List.of();
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SpringIntegration.class);
    }

    private static boolean isObjectMethod(Method m) {
        return m.getDeclaringClass() == Object.class;
    }

    private static List<SpringBridge.ServiceMethodInfo> getEndpointInfos(
            Map<String, EndpointRegistry.VaadinEndpointData> vaadinEndpoints) {
        List<SpringBridge.ServiceMethodInfo> endpointInfos = new ArrayList<>();
        for (EndpointRegistry.VaadinEndpointData endpoint : vaadinEndpoints.values()) {
            if (isInternalEndpoint(endpoint)) {
                continue;
            }
            Class<?> endpointClass = endpoint.getEndpointObject().getClass();
            Map<String, Method> methods = endpoint.getMethods();
            for (Method method : methods.values()) {
                endpointInfos.add(new SpringBridge.ServiceMethodInfo(endpointClass, method));
            }
        }
        return endpointInfos;
    }

    private static boolean isInternalEndpoint(EndpointRegistry.VaadinEndpointData endpointData) {
        String name = endpointData.getEndpointObject().getClass().getName();
        return name.startsWith("com.vaadin");
    }

    /**
     * Gets version information for Spring Boot and related libraries.
     *
     * @return version information
     */
    public static SpringBridge.VersionInfo getVersionInfo() {
        return new SpringBridge.VersionInfo(SpringBootVersion.getVersion(), SpringVersion.getVersion());
    }

    static <T> T getBean(Class<T> cls, VaadinServletContext context) {
        return getWebApplicationContext(context).getBean(cls);

    }

}