package com.vaadin.copilot.ide;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.FlowUtil;
import com.vaadin.copilot.ProjectManager;
import com.vaadin.copilot.plugins.themeeditor.ThemeModifier;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JsonUtils;

import elemental.json.JsonObject;
import elemental.json.JsonValue;

public class IdePluginCommandHandler implements CopilotCommand {

    private static final String UI_ID = "uiId";
    private final ProjectManager projectManager;

    private final CopilotIDEPlugin plugin;

    public IdePluginCommandHandler(ProjectManager projectManager) {
        this.projectManager = projectManager;
        this.plugin = CopilotIDEPlugin.getInstance();
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        try {
            if (command.equals("plugin-undo")) {
                handlePluginUndoCommand(data);
                return true;
            }

            if (command.equals("plugin-redo")) {
                handlePluginRedoCommand(data);
                return true;
            }
            if (command.equals("plugin-restart-application")) {
                handlePluginRestartApplicationCommand();
                return true;
            }

            if (command.equals("plugin-refresh") && plugin.supports(CopilotIDEPlugin.Commands.REFRESH)) {
                handlePluginRefreshCommand();
                return true;
            }

        } catch (IOException e) {
            throw new UnsupportedOperationException("Copilot Plugin operation failed", e);
        }
        return false;
    }

    private void handlePluginUndoCommand(JsonObject data) throws IOException {
        if (plugin.isActive() && plugin.supports(CopilotIDEPlugin.Commands.UNDO)) {
            List<String> files = listFiles(data);
            plugin.undo(files);
        }
    }

    private void handlePluginRedoCommand(JsonObject data) throws IOException {
        if (plugin.isActive() && plugin.supports(CopilotIDEPlugin.Commands.REDO)) {
            List<String> files = listFiles(data);
            plugin.redo(files);
        }
    }

    private void handlePluginRestartApplicationCommand() {
        if (plugin.isActive() && plugin.supports(CopilotIDEPlugin.Commands.RESTART_APPLICATION)) {
            plugin.restartApplication();
        }
    }

    private void handlePluginRefreshCommand() throws IOException {
        if (plugin.isActive()) {
            plugin.refresh();
        }
    }

    private List<String> listFiles(JsonObject data) {
        List<String> files = new ArrayList<>();
        files.addAll(JsonUtils.stream(data.getArray("files")).map(JsonValue::asString).map(this::filter).toList());
        if (data.hasKey(UI_ID)) {
            int flowUiId = (int) data.getNumber(UI_ID);
            if (flowUiId >= 0) {
                Map<ComponentTracker.Location, File> flowFiles = FlowUtil.findActiveJavaFiles(projectManager, flowUiId);
                files.addAll(flowFiles.values().stream().map(File::getAbsolutePath).toList());
            }
        }
        return files;
    }

    private String filter(String filename) {
        if (filename.equals("$THEME_EDITOR_CSS")) {
            Optional<File> themeFolder = projectManager.getThemeFolder();
            if (themeFolder.isPresent()) {
                return themeFolder.get().toPath().resolve(ThemeModifier.THEME_EDITOR_CSS).toString();
            }
        }
        return filename;
    }
}
