package com.vaadin.copilot.plugins.vaadinversionupdate;

import java.net.ConnectException;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Optional;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.copilot.ErrorHandler;
import com.vaadin.copilot.ProjectManager;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.server.Platform;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import elemental.json.Json;
import elemental.json.JsonObject;

/**
 * Handler for displaying and upgrading to new versions
 */
public class VaadinVersionUpdateHandler extends CopilotServerClient implements CopilotCommand {
    private static final String ERROR_MESSAGE = "errorMessage";
    private static final String ERROR_KEY = "error";
    private final VaadinVersionUpdate vaadinVersionUpdate;

    /**
     * Constructs a new instance of VersionCheckerHandler
     *
     * @param projectManager
     *            Project manager to access dependency management
     * @param applicationConfiguration
     *            Application configuration to access project root folder
     */
    public VaadinVersionUpdateHandler(ProjectManager projectManager,
            ApplicationConfiguration applicationConfiguration) {
        this.vaadinVersionUpdate = new VaadinVersionUpdate(projectManager, applicationConfiguration);
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-new-vaadin-versions")) {
            var reqId = data.getString(KEY_REQ_ID);
            var responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);

            Optional<String> vaadinVersion = Platform.getVaadinVersion();
            if (vaadinVersion.isEmpty()) {
                responseData.put(ERROR_KEY, true);
                responseData.put(ERROR_MESSAGE, "Unable to find vaadin version");
                return true;
            }
            boolean includePreReleases = data.hasKey("includePreReleases") && data.getBoolean("includePreReleases");
            VersionCheckRequest versionCheckRequest = new VersionCheckRequest(vaadinVersion.get(), includePreReleases);
            sendVersionRequestAndFillResponse(versionCheckRequest, responseData);
            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            return true;
        } else if (command.equals("update-vaadin-version")) {
            var reqId = data.getString(KEY_REQ_ID);
            var responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);
            try {
                vaadinVersionUpdate.updateVaadinVersion(data.getString("newVersion"),
                        data.hasKey("preRelease") && data.getBoolean("preRelease"));
                devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                        "Unable to update Vaadin Version", e);
            }
            return true;
        }
        return false;
    }

    private void sendVersionRequestAndFillResponse(VersionCheckRequest versionCheckRequest, JsonObject responseData) {
        try {
            URI uri = getQueryURI("vaadin-version");
            String json = writeAsJsonString(versionCheckRequest);
            String vaadinVersion = versionCheckRequest.version();
            HttpRequest request = buildRequest(uri, json);
            HttpResponse<String> response = getHttpClient().send(request, HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() == 200) {
                responseData.put("newVersions", JsonUtils.writeValue(response.body()));
            } else {
                if (response.statusCode() == 404) {
                    responseData.put(ERROR_MESSAGE, "Requested version[" + vaadinVersion + "] not found");
                } else if (response.statusCode() == 500) {
                    responseData.put(ERROR_MESSAGE, "Internal Server Error. Please try again later.");
                } else if (response.statusCode() == 502) {
                    responseData.put(ERROR_MESSAGE, "The server is temporarily unavailable. Please try again later.");
                } else {
                    responseData.put(ERROR_MESSAGE, "Unexpected error occurred. Please try again later.");
                }
                responseData.put(ERROR_KEY, true);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (ConnectException e) {
            getLogger().debug("Unable to connect to server", e);
            responseData.put(ERROR_KEY, true);
            responseData.put(ERROR_MESSAGE, "Unable to connect to server. Please try again later.");
        } catch (Exception e) {
            getLogger().error("Error while sending version check request", e);
            responseData.put(ERROR_KEY, true);
            responseData.put(ERROR_MESSAGE, "An unexpected error occurred while processing the request");
        }
    }

    @Override
    public boolean canBeParallelCommand(String command) {
        return command.equals("get-new-vaadin-versions");
    }

    /**
     * Request data for copilot server to fetch new versions after given version.
     *
     * @param version
     *            Version of the project
     * @param includePreReleases
     *            flag to display pre releases
     */
    public record VersionCheckRequest(String version, boolean includePreReleases) {
    }
}
