package com.vaadin.copilot.userinfo;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.ProKey;

public class UserInfoServerClient extends CopilotServerClient {

    private static Map<String, UserInfo> userInfoCache = Collections.synchronizedMap(new HashMap<>());

    public UserInfo getUserInfo(String proKey) {
        URI uri = getQueryURI("userInfo");
        UserInfoRequest requestData = new UserInfoRequest(proKey);
        String json = writeAsJsonString(requestData);
        HttpRequest request = buildRequest(uri, json);
        try {
            HttpResponse<String> response = getHttpClient().send(request, HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() != 200) {
                return null;
            }
            return readValue(response.body(), UserInfo.class);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (IOException e) {
            // no logging here as we don't want to bother developer that we
            // can't load his user info
        }

        return null;
    }

    /**
     * Requests to Copilot Server to get user information with local pro key if
     * present
     *
     * @return Returns null when pro key not exist or when request is not succeeded.
     */
    public static UserInfo getUserInfoWithLocalProKey() {
        ProKey proKey = LocalProKey.get();
        if (proKey == null) {
            return null;
        }
        UserInfo cachedUserInfo = userInfoCache.get(proKey.getProKey());
        if (cachedUserInfo == null) {
            cachedUserInfo = new UserInfoServerClient().getUserInfo(proKey.getProKey());
            userInfoCache.put(proKey.getProKey(), cachedUserInfo);
        }

        return cachedUserInfo;
    }

    /**
     * Throws a DataCannotLeaveLocalhostException if data cannot leave localhost.
     */
    public static void throwIfDataCannotLeaveLocalhost() {
        UserInfo userInfo = getUserInfoWithLocalProKey();
        boolean canLeave = (userInfo != null && userInfo.canDataLeaveLocalhost())
                || (userInfo == null && Copilot.isDevelopmentMode());
        if (!canLeave) {
            throw new DataCannotLeaveLocalhostException();
        }
    }
}
