package com.vaadin.copilot;

import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolderStrategy;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.web.SecurityFilterChain;

/**
 * Provides Spring security related helpers for copilot. Depends on Spring
 * security classes and cannot be directly imported.
 */
public class SpringSecurityIntegration {

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SpringSecurityIntegration.class);
    }

    /**
     * Checks whether Spring Security is enabled.
     *
     * @param context
     *            the Vaadin servlet context
     *
     * @return true if Spring Security is enabled, false otherwise
     */
    public static boolean isSpringSecurityEnabled(VaadinServletContext context) {
        try {
            String[] securityFilterChainBeanNames = SpringIntegration.getWebApplicationContext(context)
                    .getBeanNamesForType(SecurityFilterChain.class);
            return securityFilterChainBeanNames.length > 0;
        } catch (Exception e) {
            getLogger().trace("Exception while trying to find if Spring Security is enabled", e);
        }
        return false;
    }

    /**
     * Set active user for Spring Security.
     *
     * @param username
     *            the username
     * @param session
     *            the Vaadin session
     * @throws UsernameNotFoundException
     *             if the user was not found
     */
    public static void setActiveSpringSecurityUser(String username, VaadinSession session)
            throws UsernameNotFoundException {
        VaadinServletContext context = (VaadinServletContext) session.getService().getContext();

        SecurityContextHolderStrategy securityContextHolderStrategy = SpringIntegration
                .getBean(SecurityContextHolderStrategy.class, context);
        UserDetailsService userDetailsService = SpringIntegration.getBean(UserDetailsService.class, context);

        UserDetails targetUser = userDetailsService.loadUserByUsername(username);
        UsernamePasswordAuthenticationToken targetUserRequest = UsernamePasswordAuthenticationToken
                .authenticated(targetUser, targetUser.getPassword(), targetUser.getAuthorities());
        SecurityContext securityContext = securityContextHolderStrategy.createEmptyContext();
        securityContext.setAuthentication(targetUserRequest);
        securityContextHolderStrategy.setContext(securityContext);
        session.getSession().setAttribute("SPRING_SECURITY_CONTEXT", securityContext);
    }

}
