package com.vaadin.copilot.customcomponent;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.vaadin.copilot.FlowUtil;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.RouteHandler;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.router.RouteBaseData;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.server.VaadinSession;

import elemental.json.JsonObject;
import elemental.json.JsonType;

/**
 * Helper class for custom components
 */
public class CustomComponentHelper {
    private static final String ACTIVE_CLASS_JSON_KEY = "activeClass";
    private static final String ACTIVE_NODE_ID_JSON_KEY = "activeNodeId";

    private CustomComponentHelper() {

    }

    /**
     * Generates the response that is used in the UI
     *
     * @param allComponents
     *            Component and source location of all components that are found in
     *            the UI
     * @param componentsInProject
     *            Components that have source in the project.
     * @return response that contains required metadata for the custom component
     *         feature
     */
    public static CustomComponentResponseData generateResponse(VaadinSession vaadinSession,
            Map<Component, ComponentTypeAndSourceLocation> allComponents, List<Component> componentsInProject) {

        Map<Integer, ComponentInfoForCustomComponentSupport> allComponentsInfoForCustomComponentSupport = new HashMap<>();
        CustomComponentResponseData responseData = new CustomComponentResponseData(
                allComponentsInfoForCustomComponentSupport);
        componentsInProject.stream().filter(component -> CustomComponents.isCustomComponent(component.getClass()))
                .forEach(component -> {
                    Integer nodeId = FlowUtil.getNodeId(component);

                    CustomComponent customComponent = CustomComponents.getCustomComponentInfo(component).orElseThrow();
                    String activeLevel;
                    if (CustomComponent.Type.IN_PROJECT.equals(customComponent.getType())) {
                        activeLevel = ((CustomComponentInProject) customComponent).sourceFile().getName();
                    } else {
                        activeLevel = "External [" + customComponent.componentClass().getSimpleName() + "]";
                    }
                    String filePath = null;
                    if (customComponent instanceof CustomComponentInProject customComponentInProject) {
                        filePath = customComponentInProject.sourceFile().getPath();
                    }
                    CustomComponentInstanceInfo customComponentInstanceInfo = new CustomComponentInstanceInfo(
                            customComponent.getType(), activeLevel, customComponent.getChildAddableMethods(),
                            customComponent.componentClass().getName(), filePath);
                    allComponentsInfoForCustomComponentSupport.put(nodeId, customComponentInstanceInfo);
                });

        componentsInProject.forEach(component -> {
            ComponentTypeAndSourceLocation typeAndSourceLocation = allComponents.get(component);
            Integer nodeId = FlowUtil.getNodeId(component);
            Optional<ComponentTracker.Location> locationInProject = typeAndSourceLocation.createLocationInProject();
            String filePath = null;
            if (locationInProject.isPresent()) {
                filePath = ProjectFileManager.get().getSourceFile(locationInProject.get()).getPath();
            }
            if (!allComponentsInfoForCustomComponentSupport.containsKey(nodeId)) {
                allComponentsInfoForCustomComponentSupport.put(nodeId, new ComponentInfoForCustomComponentSupport());
            }
            boolean childOfCustomComponent = FlowUtil.testAncestors(vaadinSession, component,
                    ancestor -> CustomComponents.isCustomComponent(ancestor.getClass()));
            allComponentsInfoForCustomComponentSupport.get(nodeId).setCreateLocationPath(filePath);
            allComponentsInfoForCustomComponentSupport.get(nodeId).setChildOfCustomComponent(childOfCustomComponent);
        });

        Integer routeNodeId = getRouteNodeId(componentsInProject, vaadinSession);
        if (allComponentsInfoForCustomComponentSupport.containsKey(routeNodeId)) {
            allComponentsInfoForCustomComponentSupport.get(routeNodeId).setRouteView(true);
        }
        return responseData;
    }

    private static Integer getRouteNodeId(Collection<Component> components, VaadinSession vaadinSession) {
        List<RouteData> serverRoutes = RouteHandler.getServerRoutes(vaadinSession);
        if (serverRoutes == null) {
            return null;
        }
        List<? extends Class<? extends Component>> navigationTargets = serverRoutes.stream()
                .map(RouteBaseData::getNavigationTarget).toList();
        Optional<Component> first = components.stream().filter(comp -> navigationTargets.contains(comp.getClass()))
                .findFirst();
        return first.map(FlowUtil::getNodeId).orElse(null);

    }

    /**
     * Checks if the given component is drilled down
     *
     * @param vaadinSession
     *            Vaadin session to access component
     * @param data
     *            Sent data from client. It requires
     *            {@link CustomComponentHelper#ACTIVE_CLASS_JSON_KEY} and
     *            {@link CustomComponentHelper#ACTIVE_NODE_ID_JSON_KEY}
     * @param component
     *            JSON data that contains uiId and nodeId
     * @return true if drilled down component, false otherwise
     */
    public static boolean isDrilledDownComponent(VaadinSession vaadinSession, JsonObject data, JsonObject component) {
        if (component == null) {
            return false;
        }
        return isDrilledDownComponent(vaadinSession, data, (int) component.getNumber("uiId"),
                (int) component.getNumber("nodeId"));
    }

    /**
     * Checks if the given component is drilled down.
     *
     * @param vaadinSession
     *            Vaadin session to access component
     * @param data
     *            Sent data from client. It requires
     *            {@link CustomComponentHelper#ACTIVE_CLASS_JSON_KEY} and
     *            {@link CustomComponentHelper#ACTIVE_NODE_ID_JSON_KEY}
     * @param uiId
     *            uiId
     * @param nodeId
     *            nodeId of the given component
     * @return true if drilled down component, false otherwise
     */
    public static boolean isDrilledDownComponent(VaadinSession vaadinSession, JsonObject data, int uiId, int nodeId) {
        if (!data.hasKey(ACTIVE_CLASS_JSON_KEY) || data.get(ACTIVE_CLASS_JSON_KEY).getType().equals(JsonType.NULL)) {
            return false;
        }
        if (!data.hasKey(ACTIVE_NODE_ID_JSON_KEY)
                || data.get(ACTIVE_NODE_ID_JSON_KEY).getType().equals(JsonType.NULL)) {
            return false;
        }
        String activeClass = data.getString(ACTIVE_CLASS_JSON_KEY);
        int activeNodeId = (int) data.getNumber(ACTIVE_NODE_ID_JSON_KEY);
        if (activeNodeId != nodeId) {
            return false;
        }

        Optional<Component> componentByNodeIdAndUiId = FlowUtil.findComponentByNodeIdAndUiId(vaadinSession, nodeId,
                uiId);
        if (componentByNodeIdAndUiId.isEmpty()) {
            return false;
        }
        if (!componentByNodeIdAndUiId.get().getClass().getName().equals(activeClass)) {
            return false;
        }
        return true;
    }
}
