package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.util.Optional;

import com.vaadin.copilot.customcomponent.CustomComponent;
import com.vaadin.flow.component.Component;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;

/**
 * Represents metadata information about a Vaadin {@link Component}.
 * <p>
 * This record aggregates various details about a component's structure,
 * creation, and attachment lifecycle, including information about whether it is
 * a container composite, anonymous, a return value from a method, created
 * inside a loop, and associated custom metadata.
 * </p>
 *
 * @param type
 *            the {@link Class} type of the component
 * @param routeConstructor
 *            the {@link ConstructorDeclaration} associated with the component's
 *            creation
 * @param containerComposite
 *            {@code true} if the component is a container composite
 * @param isAnonymousComponent
 *            {@code true} if the component is an anonymous class
 * @param isReturnValue
 *            {@code true} if the component is the return value of a method
 * @param createdInLoop
 *            {@code true} if the component is created inside a loop
 * @param customComponentInfo
 *            optional metadata about custom components
 * @param componentCreateInfoOptional
 *            optional metadata about the component's creation process
 * @param componentAttachInfoOptional
 *            optional metadata about the component's attachment lifecycle
 */

public record ComponentInfo(Class<? extends Component> type, ConstructorDeclaration routeConstructor,
        boolean containerComposite, boolean isAnonymousComponent, boolean isReturnValue, boolean createdInLoop,
        boolean createAndAttachLocationsAreInSameFile, Optional<CustomComponent> customComponentInfo,
        Optional<ComponentCreateInfo> componentCreateInfoOptional,
        Optional<ComponentAttachInfo> componentAttachInfoOptional) {

    public AttachExpression getAttachCallInSameFileOrThrow() {
        if (componentAttachInfoOptional.isEmpty()) {
            return null;
        }
        ComponentAttachInfo componentAttachInfo = componentAttachInfoOptional.get();
        if (!createAndAttachLocationsAreInSameFile) {
            throw new IllegalArgumentException(
                    "Create and attach location are in different files. This operation is not supported at the moment");
        }
        return componentAttachInfo.getAttachCall();
    }

    public Optional<BlockStmt> getComponentAttachScopeOrThrowIfDifferentFile() {
        if (componentAttachInfoOptional.isEmpty()) {
            return Optional.empty();
        }
        ComponentAttachInfo componentAttachInfo = componentAttachInfoOptional.get();
        if (!createAndAttachLocationsAreInSameFile) {
            throw new IllegalArgumentException(
                    "Create and attach location are in different files. This operation is not supported at the moment");
        }
        return componentAttachInfo.getComponentAttachScope();
    }

    public ComponentCreateInfo getCreateInfoOrThrow() {
        return componentCreateInfoOptional
                .orElseThrow(() -> new IllegalStateException("Component create info is not present"));
    }

    public File getCreateLocationFileOrThrowIfNull() {
        return componentCreateInfoOptional
                .orElseThrow(() -> new IllegalArgumentException("Create location rewriter is null")).getFile();
    }

    public CompilationUnit getCreateLocationCompilationUnitOrThrowIfNull() {
        return componentCreateInfoOptional
                .orElseThrow(() -> new IllegalArgumentException("Create location compilation unit is null!"))
                .getJavaSource().getCompilationUnit();
    }

    public CompilationUnit getAttachLocationCompilationUnitOrThrowIfNull() {
        return componentAttachInfoOptional
                .orElseThrow(() -> new IllegalArgumentException("Attach location compilation unit is null!"))
                .getJavaSource().getCompilationUnit();
    }

    /**
     * Gets the variable declarator of the component, either for the local variable
     * or field.
     *
     * @return the variable declarator of the component, or null if not found
     */
    public VariableDeclarator getVariableDeclarator() {
        ComponentCreateInfo componentCreateInfo = componentCreateInfoOptional.get();

        if (componentCreateInfo.getLocalVariableDeclarator() != null) {
            return componentCreateInfo.getLocalVariableDeclarator();
        } else if (componentCreateInfo.getFieldDeclaration() != null) {
            return componentCreateInfo.getFieldDeclaration().getVariables().stream()
                    .filter(variable -> variable.getNameAsString().equals(componentCreateInfo.getFieldName()))
                    .findFirst().orElse(null);
        }

        return null;
    }

    public String typeWithGenerics() {
        VariableDeclarator variableDeclarator = getVariableDeclarator();
        String typeFromClass = type().getSimpleName();
        if (variableDeclarator == null) {
            return typeFromClass;
        }
        if (!variableDeclarator.getType().isVarType()) {
            return variableDeclarator.getType().toString();
        }
        return variableDeclarator.getInitializer().filter(Expression::isObjectCreationExpr)
                .map(ObjectCreationExpr.class::cast).map(a -> a.getType().toString()).orElse(typeFromClass);
    }
}
