package com.vaadin.copilot.javarewriter;

import java.util.Optional;

import com.vaadin.copilot.customcomponent.CustomComponent;
import com.vaadin.flow.component.Component;

import com.github.javaparser.ast.body.ConstructorDeclaration;

/**
 * Builder pattern for {@link ComponentInfo} class
 */
public class ComponentInfoBuilder {

    private Class<? extends Component> type;
    private ConstructorDeclaration routeConstructor;
    private boolean containerComposite;
    private boolean isAnonymousComponent;
    private boolean isReturnValue;
    private boolean createdInLoop;
    private boolean classSource;
    private boolean createAndAttachLocationsAreInSameFile;
    private Optional<CustomComponent> customComponentInfo = Optional.empty();
    private Optional<ComponentCreateInfo> componentCreateInfoOptional = Optional.empty();
    private Optional<ComponentAttachInfo> componentAttachInfoOptional = Optional.empty();

    /**
     * Sets the type of the component.
     *
     * @param type
     *            the {@link Class} representing the component type; must not be
     *            {@code null}.
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder type(Class<? extends Component> type) {
        this.type = type;
        return this;
    }

    /**
     * Sets the constructor of the route where the component is created.
     *
     * @param routeConstructor
     *            the {@link ConstructorDeclaration} associated with the component
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder routeConstructor(ConstructorDeclaration routeConstructor) {
        this.routeConstructor = routeConstructor;
        return this;
    }

    /**
     * Specifies whether the component is part of a container composite structure.
     *
     * @param containerComposite
     *            {@code true} if it is a container composite, {@code false}
     *            otherwise
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder containerComposite(boolean containerComposite) {
        this.containerComposite = containerComposite;
        return this;
    }

    /**
     * Sets whether the component is an anonymous component (i.e., without a named
     * type).
     *
     * @param isAnonymousComponent
     *            {@code true} if it is an anonymous component, {@code false}
     *            otherwise
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder isAnonymousComponent(boolean isAnonymousComponent) {
        this.isAnonymousComponent = isAnonymousComponent;
        return this;
    }

    /**
     * Specifies whether the component is used as a return value from a method or
     * constructor.
     *
     * @param isReturnValue
     *            {@code true} if the component is a return value, {@code false}
     *            otherwise
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder isReturnValue(boolean isReturnValue) {
        this.isReturnValue = isReturnValue;
        return this;
    }

    /**
     * Sets whether the component is created inside a loop.
     *
     * @param createdInLoop
     *            {@code true} if created inside a loop, {@code false} otherwise
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder createdInLoop(boolean createdInLoop) {
        this.createdInLoop = createdInLoop;
        return this;
    }

    /**
     * Sets custom information associated with the component.
     *
     * @param customComponentInfo
     *            a {@link CustomComponent} instance with additional component
     *            metadata, or {@code null} if none
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder customComponentInfo(CustomComponent customComponentInfo) {
        this.customComponentInfo = Optional.ofNullable(customComponentInfo);
        return this;
    }

    /**
     * Sets the creation information associated with the component.
     *
     * @param componentCreateInfo
     *            a {@link ComponentCreateInfo} instance containing creation
     *            details, or {@code null} if none
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder componentCreateInfo(ComponentCreateInfo componentCreateInfo) {
        this.componentCreateInfoOptional = Optional.ofNullable(componentCreateInfo);
        return this;
    }

    /**
     * Sets the attachment information associated with the component.
     *
     * @param componentAttachInfo
     *            a {@link ComponentAttachInfo} instance containing attachment
     *            details, or {@code null} if none
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder componentAttachInfo(ComponentAttachInfo componentAttachInfo) {
        this.componentAttachInfoOptional = Optional.ofNullable(componentAttachInfo);
        return this;
    }

    /**
     * Sets whether component create location and attach location are in the same
     * file.
     *
     * @param createAndAttachLocationsAreInSameFile
     *            true when both refers to same file, {@code false} otherwise.
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder createAndAttachLocationsAreInSameFile(boolean createAndAttachLocationsAreInSameFile) {
        this.createAndAttachLocationsAreInSameFile = createAndAttachLocationsAreInSameFile;
        return this;
    }

    /**
     * Sets whether the component indicates the whole class file rather than
     * instances in the class
     *
     * @param classSource
     *            {@code true} if component info refers to class, false otherwise.
     * @return the updated {@code ComponentInfoBuilder} instance
     */
    public ComponentInfoBuilder classSource(boolean classSource) {
        this.classSource = classSource;
        return this;
    }

    /**
     * Returns class source value
     *
     * @return true if component info refers to class, false otherwise.
     */
    public boolean isClassSource() {
        return classSource;
    }

    /**
     * Returns the {@link ComponentCreateInfo} if present, or throws an exception if
     * absent.
     *
     * @return the {@link ComponentCreateInfo} associated with the component
     * @throws IllegalStateException
     *             if the create info is not present
     */
    public ComponentCreateInfo getCreateInfoOrThrow() {
        return componentCreateInfoOptional
                .orElseThrow(() -> new IllegalStateException("Component info is not created"));
    }

    /**
     * Builds a new {@link ComponentInfo} instance based on the values set in this
     * builder.
     *
     * @return a new {@link ComponentInfo} instance
     */
    public ComponentInfo build() {
        return new ComponentInfo(type, routeConstructor, containerComposite, isAnonymousComponent, isReturnValue,
                createdInLoop, createAndAttachLocationsAreInSameFile, customComponentInfo, componentCreateInfoOptional,
                componentAttachInfoOptional);
    }
}
