package com.vaadin.copilot;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;
import com.vaadin.pro.licensechecker.ProKey;

import elemental.json.JsonObject;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import io.netty.buffer.ByteBufAllocator;
import io.netty.handler.codec.http.HttpMethod;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Disposable;
import reactor.core.publisher.Mono;
import reactor.netty.ByteBufMono;

/**
 * Base class for Copilot Server API clients
 */
public class CopilotServerClient {

    protected static final String SERVER_URL_ENV = "copilot.serverBaseUrl";
    private static final String SERVER_PORT_ENV = "COPILOT_SERVER_PORT";
    private static String serverBaseUrlCache = null;

    private final HttpClient httpClient;

    private final ObjectMapper objectMapper;

    /**
     * Constructor initializing HttpClient and ObjectMapper
     */
    public CopilotServerClient() {
        this.httpClient = HttpClient.newBuilder().version(HttpClient.Version.HTTP_2)
                .followRedirects(HttpClient.Redirect.NORMAL).build();
        this.objectMapper = new ObjectMapper();
        this.objectMapper.registerModule(new JavaTimeModule());
        this.objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    public HttpClient getHttpClient() {
        return httpClient;
    }

    public HttpRequest buildRequest(URI uri, String json) {
        return HttpRequest.newBuilder().uri(uri).POST(HttpRequest.BodyPublishers.ofString(json))
                .header("Content-Type", "application/json; charset=utf-8").timeout(Duration.ofSeconds(120)).build();
    }

    public String writeAsJsonString(Object obj) {
        try {
            return objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    public <T> T readValue(String string, Class<T> clazz) {
        try {
            return objectMapper.readValue(string, clazz);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    public URI getQueryURI(String path) {
        try {
            return new URI(getServerBaseUrl() + path);
        } catch (URISyntaxException e) {
            throw new IllegalStateException("Invalid server configuration, server uri is wrong", e);
        }
    }

    protected String getServerBaseUrl() {
        if (serverBaseUrlCache != null) {
            return serverBaseUrlCache;
        }
        serverBaseUrlCache = System.getenv(SERVER_URL_ENV);
        if (serverBaseUrlCache != null) {
            return serverBaseUrlCache;
        }

        if (Copilot.isDevelopmentMode()) {
            // Try a localhost server
            try {
                Integer serverPort = Optional.ofNullable(System.getenv(SERVER_PORT_ENV)).map(Integer::parseInt)
                        .orElse(8081);
                String localhostUrl = "http://localhost:%s/v1/".formatted(serverPort);
                String statusUrl = localhostUrl + "actuator/health";

                HttpRequest request = HttpRequest.newBuilder().uri(new URI(statusUrl)).timeout(Duration.ofSeconds(1))
                        .build();
                HttpResponse<String> response = httpClient.send(request, HttpResponse.BodyHandlers.ofString());
                if (response.statusCode() == 200) {
                    getLogger().info("Using the local server for development.");
                    serverBaseUrlCache = localhostUrl;
                    return serverBaseUrlCache;
                }
            } catch (InterruptedException e) {
                getLogger().warn("Interrupted while trying to reach the local server.", e);
                Thread.currentThread().interrupt();
            } catch (Exception e) {
                getLogger().warn(
                        "No local server found, as this exception occurred while trying to reach the local server: {}.",
                        e.getClass());
            }
            getLogger().info("Using the staging server for development.");
            serverBaseUrlCache = "https://copilot.stg.vaadin.com/v1/";
            return serverBaseUrlCache;
        } else {
            serverBaseUrlCache = "https://copilot.vaadin.com/v1/";
            return serverBaseUrlCache;
        }
    }

    public <REQ, RESP> void sendCopilotRequest(String path, REQ request, Class<RESP> responseType,
            Consumer<RESP> responseHandler, DevToolsInterface devToolsInterface, String command, JsonObject respData) {
        URI uri = getQueryURI(path);
        HttpRequest httpRequest = buildRequest(uri, writeAsJsonString(request));
        CompletableFuture<HttpResponse<String>> resp = getHttpClient().sendAsync(httpRequest,
                HttpResponse.BodyHandlers.ofString());

        resp.whenComplete((httpResponse, throwable) -> {
            if (throwable != null || httpResponse.statusCode() != 200) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, "Internal error in service",
                        throwable);
                return;
            }
            RESP response = readValue(httpResponse.body(), responseType);
            responseHandler.accept(response);
        }).whenComplete((httpResponse, throwable) -> {
            if (throwable != null) {
                // This is an exception thrown in the response handler
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData,
                        "Internal error in response handler", throwable);
            }
        });

    }

    public Disposable sendReactive(String path, String json) {
        return reactor.netty.http.client.HttpClient.create()
                .headers(headers -> headers.set("Content-Type", "application/json; charset=utf-8"))
                .request(HttpMethod.POST)
                .send(ByteBufMono.fromString(Mono.just(json), StandardCharsets.UTF_8, ByteBufAllocator.DEFAULT))
                .uri(getQueryURI(path)).response().subscribe();
    }

    public record AccessControlData(String proKey, String machineId, String copilotVersion) {
        public static AccessControlData create() {
            ProKey proKey = LocalProKey.get();

            return new AccessControlData(proKey == null ? null : proKey.getProKey(), MachineId.get(),
                    CopilotVersion.getVersion());
        }
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
