package com.vaadin.copilot;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.analytics.AnalyticsClient;
import com.vaadin.copilot.userinfo.UserInfo;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import elemental.json.JsonType;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ErrorHandler extends CopilotCommand {

    private static final Logger logger = LoggerFactory.getLogger(ErrorHandler.class);

    public static final String EXCEPTION_MESSAGE_KEY = "exceptionMessage";
    public static final String EXCEPTION_STACKTRACE_KEY = "exceptionStacktrace";
    private static final String DETAILS_JSON_KEY = "details";
    private final CopilotServerClient copilotServerClient = new CopilotServerClient();

    public static JsonArray toJson(Throwable e) {
        StringWriter sw = new StringWriter();
        try (PrintWriter pw = new PrintWriter(sw)) {
            e.printStackTrace(pw);
        }
        // We don't need the stack trace after
        // "com.vaadin.base.devserver.DebugWindowConnection.onMessage" as that
        // is outside Copilot
        // But we want all "causes" parts of the stack trace
        String[] stack = sw.toString().split("\n");

        AtomicBoolean include = new AtomicBoolean(true);
        Stream<String> filtered = Arrays.stream(stack).filter(s -> {
            if (s.contains("com.vaadin.base.devserver.DebugWindowConnection.onMessage")) {
                include.set(false);
            }
            if (!s.startsWith("\t")) {
                include.set(true);
            }
            return include.get();
        });
        return filtered.map(Json::create).collect(JsonUtils.asArray());
    }

    public static void setErrorMessage(JsonObject respData, String error) {
        setError(respData, error, null);
    }

    public static void setError(JsonObject respData, String errorMessage, Throwable e) {
        JsonObject error = Json.createObject();
        error.put("message", errorMessage);

        if (e != null) {
            String exceptionMessage = e.getMessage();
            if (exceptionMessage == null) {
                exceptionMessage = "";
            }
            error.put(EXCEPTION_MESSAGE_KEY, exceptionMessage);
            error.put(EXCEPTION_STACKTRACE_KEY, ErrorHandler.toJson(e));
            if (e instanceof SuggestRestartException) {
                error.put("suggestRestart", true);
            }
            if (Copilot.isDevelopmentMode()) {
                logger.error("Stack trace because in Copilot development mode", e);
            }
        }
        respData.put("error", error);
    }

    public static void sendErrorResponse(DevToolsInterface devToolsInterface, String command, JsonObject responseData,
            String error, Throwable e) {
        ErrorHandler.setError(responseData, error, e);
        devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
    }

    public ErrorHandler() {
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if ("error".equals(command)) {
            if (!allowedToSendErrors()) {
                return true;
            }

            String proKey = LocalProKey.get().getProKey();
            String machineId = MachineId.get();
            URI uri = copilotServerClient.getQueryURI("errors");
            String details = data.hasKey(DETAILS_JSON_KEY)
                    && !data.get(DETAILS_JSON_KEY).getType().equals(JsonType.NULL) ? data.getString(DETAILS_JSON_KEY)
                            : "";
            ErrorsRequest trackingRequest = new ErrorsRequest(machineId, proKey, data.getString("message"),
                    Map.of(DETAILS_JSON_KEY, details, "versions", data.getString("versions")));
            String json = copilotServerClient.writeAsJsonString(trackingRequest);
            HttpRequest request = copilotServerClient.buildRequest(uri, json);
            copilotServerClient.getHttpClient().sendAsync(request, HttpResponse.BodyHandlers.ofString());
            AnalyticsClient.getInstance().track("error", Map.of("message", data.getString("message")));
            return true;
        }
        return false;
    }

    private boolean allowedToSendErrors() {
        if (!AnalyticsClient.getInstance().isEnabled()) {
            return false;
        }

        UserInfo userInfo = UserInfoServerClient.getUserInfoWithLocalProKey();
        if (userInfo == null) {
            return false;
        } else if (userInfo.copilotProjectCannotLeaveLocalhost()) {
            return false;
        }
        return MachineConfiguration.get().isSendErrorReportsAllowed();
    }
}