package com.vaadin.copilot;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import com.vaadin.flow.component.Component;

import com.googlecode.gentyref.GenericTypeReflector;

public class JavaReflectionUtil {

    public record TypeInfo(String typeName, List<TypeInfo> typeParameters) {
    }

    public record ParameterTypeInfo(String name, TypeInfo type) {
    }

    public static List<ParameterTypeInfo> getParameterTypes(Method method, Class<?> cls) {
        Type[] parameterTypes;
        try {
            parameterTypes = GenericTypeReflector.getExactParameterTypes(method, cls);
        } catch (Exception e) {
            parameterTypes = method.getGenericParameterTypes();
        }
        List<ParameterTypeInfo> parameterTypeInfo = new ArrayList<>();
        for (int i = 0; i < parameterTypes.length; i++) {
            String name = method.getParameters()[i].getName();
            TypeInfo typeInfo = getTypeInfo(parameterTypes[i]);
            parameterTypeInfo.add(new ParameterTypeInfo(name, typeInfo));
        }
        return parameterTypeInfo;
    }

    private static TypeInfo getTypeInfo(Type type) {
        if (type instanceof ParameterizedType parameterizedReturnType) {
            Type rawType = parameterizedReturnType.getRawType();
            String typeName;
            if (rawType instanceof Class cls) {
                typeName = getClassName(cls);
            } else {
                typeName = rawType.getTypeName();
            }
            List<TypeInfo> typeParameters = Arrays.stream(parameterizedReturnType.getActualTypeArguments())
                    .map(JavaReflectionUtil::getTypeInfo).toList();
            return new TypeInfo(typeName, typeParameters);
        } else if (type instanceof Class<?> classType) {
            return new TypeInfo(getClassName(classType), new ArrayList<>());
        } else if (type instanceof TypeVariable<?> typeVariable) {
            return new TypeInfo(typeVariable.getName(), new ArrayList<>());
        }
        return new TypeInfo("???", new ArrayList<>());
    }

    public static TypeInfo getReturnType(Method method, Class<?> cls) {
        Type returnType;
        try {
            returnType = GenericTypeReflector.getExactReturnType(method, cls);
        } catch (Exception e) {
            returnType = method.getGenericReturnType();
        }
        return getTypeInfo(returnType);
    }

    public static String getClassName(Class<?> aClass) {
        return Optional.ofNullable(aClass.getCanonicalName()).orElse(aClass.getName());
    }

    /**
     * Checks whether the given method parameter is (or extends) a {@link Component}
     * type, either directly or through inheritance. If the parameter is an array,
     * the component type of the array is used.
     *
     * <p>
     * This method walks up the class hierarchy to determine if the parameter type
     * (or its component type in case of an array) is a subclass of
     * {@code Component}.
     *
     * @param parameter
     *            the method parameter to inspect
     * @return {@code true} if the parameter type or its component type is a
     *         subclass of {@code Component}, {@code false} otherwise
     */
    public static boolean isParamFlowComponentType(Parameter parameter) {
        Class<?> type;
        if (parameter.getType().isArray()) {
            type = parameter.getType().componentType();
        } else {
            type = parameter.getType();
        }
        while (type != null) {
            if (type.equals(Component.class)) {
                return true;
            }
            type = type.getSuperclass();
        }
        return false;
    }
}
