package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.vaadin.base.devserver.stats.ProjectHelpers;
import com.vaadin.flow.server.frontend.installer.ArchiveExtractionException;
import com.vaadin.flow.server.frontend.installer.DefaultArchiveExtractor;
import com.vaadin.flow.server.frontend.installer.DownloadException;
import com.vaadin.open.OSUtils;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

/** Utility class for downloading and referencing a JetBrains Runtime. */
public final class JetbrainsRuntimeUtil {

    private static final String TAR_GZ = ".tar.gz";
    private static final String JETBRAINS_GITHUB_RELEASES_PAGE = "https://api.github.com/repos/JetBrains/JetBrainsRuntime/releases";
    private final ObjectMapper objectMapper = new ObjectMapper();

    /** Describes a GitHub release with a body. */
    record GitHubReleaseWithBody(String body) {
    }

    /** Describes a GitHub release. */
    record GitHubRelease(int id, String name, String tag_name,
            boolean prerelease) implements Comparable<GitHubRelease> {

        @Override
        public int compareTo(GitHubRelease o) {
            return compareNumerically(tag_name, o.tag_name);
        }

        private int compareNumerically(String s1, String s2) {
            int commonLength = Math.min(s1.length(), s2.length());
            for (int i = 0; i < commonLength; i++) {
                char c1 = s1.charAt(i);
                char c2 = s2.charAt(i);
                if (c1 == c2) {
                    continue;
                }
                return Character.compare(c1, c2);
            }
            if (s1.length() > commonLength) {
                return 1;
            } else if (s2.length() > commonLength) {
                return -1;
            }
            return 0;
        }

    }

    /** Describes a version of JetBrains Runtime SDK that could be downloaded. */
    record JBRSdkInfo(String arch, String sdkType, String url) {
    }

    /**
     * Creates a new instance of JetbrainsRuntimeUtil.
     *
     */
    public JetbrainsRuntimeUtil() {
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    private static String getArchitecture() {
        return System.getProperty("os.arch");
    }

    /**
     * Unpacks the given JetBrains Runtime archive into a folder.
     *
     * @param jbrArchive
     *            the archive to unpack
     * @param statusUpdater
     *            the consumer to pass status information to
     * @return the folder where the archive was unpacked
     * @throws IOException
     *             if an I/O error occurs
     * @throws ArchiveExtractionException
     *             if the archive cannot be extracted
     */
    public File unpackJbr(File jbrArchive, Consumer<String> statusUpdater)
            throws IOException, ArchiveExtractionException {
        // Unpack the archive
        if (!jbrArchive.getName().endsWith(TAR_GZ)) {
            throw new IOException("Unexpected file format: " + jbrArchive.getName());
        }

        File folder = new File(jbrArchive.getParentFile(), jbrArchive.getName().replace(TAR_GZ, ""));
        if (folder.exists()) {
            File[] files = folder.listFiles();
            if (files == null || files.length == 0) {
                Files.delete(folder.toPath());
            } else {
                statusUpdater.accept("Using JetBrains already in " + folder.getAbsolutePath());
                return folder;
            }
        }
        statusUpdater.accept("Extracting " + jbrArchive.getAbsolutePath() + " into " + folder.getAbsolutePath());

        new DefaultArchiveExtractor().extract(jbrArchive, jbrArchive.getParentFile());
        statusUpdater.accept("Extraction complete");

        return folder;
    }

    private File getJavaHome(File jdkFolder) {
        if (OSUtils.isMac()) {
            return jdkFolder.toPath().resolve("Contents").resolve("Home").toFile();
        }
        return jdkFolder;
    }

    /**
     * Returns the location inside the JDK where HotswapAgent should be placed.
     *
     * @param jdkFolder
     *            the JDK folder
     * @return the location inside the JDK where HotswapAgent should be placed
     */
    public File getHotswapAgentLocation(File jdkFolder) {
        return new File(new File(new File(getJavaHome(jdkFolder), "lib"), "hotswap"), "hotswap-agent.jar");
    }

    /**
     * Returns the location of the Java executable inside the JDK.
     *
     * @param jdkFolder
     *            the JDK folder
     * @return the location of the Java executable inside the JDK
     */
    public File getJavaExecutable(File jdkFolder) {
        String bin = OSUtils.isWindows() ? "java.exe" : "java";
        return new File(new File(getJavaHome(jdkFolder), "bin"), bin);
    }

    /**
     * Downloads the latest JetBrains Runtime and returns the downloaded file.
     *
     * <p>
     * The downloaded file will be placed in the Vaadin home directory under the
     * "jdk" folder. If the file already exists, it will not be downloaded again.
     *
     * <p>
     * If no suitable download is found, an empty optional is returned.
     *
     * @param statusUpdater
     *            the consumer to pass status information to
     * @return the downloaded file, if any
     * @throws IOException
     *             if an I/O error occurs
     * @throws URISyntaxException
     *             if there is an internal error with url handling
     * @throws DownloadException
     *             if the download fails
     */
    public Optional<File> downloadLatestJBR(Consumer<String> statusUpdater)
            throws IOException, URISyntaxException, DownloadException {
        statusUpdater.accept("Finding JetBrains Runtime download location");

        // Fetch the latest release from the given URL
        GitHubRelease latest = findLatestJBRRelease();
        Optional<URL> downloadUrl = findJBRDownloadUrl(latest);
        if (downloadUrl.isPresent()) {
            URL url = downloadUrl.get();
            String filename = getFilename(url);
            File target = new File(new File(ProjectHelpers.resolveVaadinHomeDirectory(), "jdk"), filename);
            File folder = target.getParentFile();
            if (!folder.exists() && !folder.mkdirs()) {
                throw new IOException("Unable to create " + folder.getAbsolutePath());
            }

            downloadIfNotPresent(url, target, statusUpdater);
            return Optional.of(target);
        }
        return Optional.empty();
    }

    private void downloadIfNotPresent(URL url, File target, Consumer<String> statusUpdater)
            throws URISyntaxException, DownloadException {
        if (target.exists() && target.length() > 0) {
            statusUpdater.accept("JetBrains Runtime already downloaded into " + target.getAbsolutePath());
            return;
        }
        statusUpdater.accept("Downloading JetBrains Runtime from " + url);
        Downloader.downloadFile(url, target, (bytesTransferred, totalBytes, progress) -> statusUpdater
                .accept(HotswapDownloadHandler.PROGRESS_PREFIX + progress));
        statusUpdater.accept("Downloaded JetBrains Runtime into " + target.getAbsolutePath());
    }

    private String getFilename(URL url) {
        return url.getFile().replaceAll(".*/", "");
    }

    private Optional<URL> findJBRDownloadUrl(GitHubRelease latest) throws URISyntaxException, IOException {
        GitHubReleaseWithBody release = objectMapper.readValue(
                new URI("https://api.github.com/repos/JetBrains/JetBrainsRuntime/releases/" + latest.id).toURL(),
                GitHubReleaseWithBody.class);
        Optional<JBRSdkInfo> sdk = findCorrectReleaseForArchitecture(release.body);
        if (sdk.isPresent()) {
            return Optional.of(new URL(sdk.get().url()));
        }
        return Optional.empty();
    }

    private GitHubRelease findLatestJBRRelease() throws URISyntaxException, IOException {
        // Right now, JBR 17 is marked as "latest" so we sort based on tag names
        // instead

        TypeReference<List<GitHubRelease>> listOfGithubReleases = new TypeReference<List<GitHubRelease>>() {
        };
        List<GitHubRelease> res = objectMapper.readValue(new URI(JETBRAINS_GITHUB_RELEASES_PAGE).toURL(),
                listOfGithubReleases);
        List<GitHubRelease> relevant = res.stream().filter(r -> !r.prerelease).sorted().toList();

        return relevant.get(relevant.size() - 1);
    }

    private Optional<JBRSdkInfo> findCorrectReleaseForArchitecture(String body) {
        Map<String, JBRSdkInfo> jbrSdks = findAllJbrSdks(body);
        String key = getDownloadKey();
        return Optional.ofNullable(jbrSdks.get(key));
    }

    /**
     * Returns the key to use for downloading the correct JBR SDK for the current
     * architecture.
     *
     * @return the key to use for downloading
     */
    public String getDownloadKey() {
        String jvmArch = getArchitecture();
        String prefix;
        if (OSUtils.isMac()) {
            prefix = "osx";
        } else if (OSUtils.isWindows()) {
            prefix = "windows";
        } else {
            prefix = "linux";
        }
        String suffix;
        if ("aarch64".equals(jvmArch)) {
            suffix = "aarch64";
        } else if ("x86".equals(jvmArch)) {
            suffix = "x86";
        } else {
            suffix = "x64";
        }
        return prefix + "-" + suffix;
    }

    private static Map<String, JBRSdkInfo> findAllJbrSdks(String body) {
        String[] lines = body.replace("\r", "").split("\n");
        List<JBRSdkInfo> sdks = Arrays.stream(lines).map(line -> {
            String[] parts = line.split("\\|");
            if (parts.length < 4) {
                return null;
            }
            String arch = parts[1].trim();
            String sdkType = parts[2].replace("*", "").trim();
            String url = parts[3].replaceAll("\\[.*]", "").replaceAll("[()]", "").trim();
            return new JBRSdkInfo(arch, sdkType, url);
        }).filter(Objects::nonNull).toList();
        return sdks.stream().filter(jbrSdkInfo -> jbrSdkInfo.sdkType.equals("JBRSDK"))
                .filter(jbrSdkInfo -> jbrSdkInfo.url.endsWith(TAR_GZ))
                .collect(Collectors.toMap(sdkInfo -> sdkInfo.arch, sdkInfo -> sdkInfo));
    }
}
