package com.vaadin.copilot.javarewriter.custom;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.FlowComponentQuirks;
import com.vaadin.copilot.javarewriter.InsertionPoint;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;

import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.IntegerLiteralExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.type.ClassOrInterfaceType;

/**
 * Handlers for Dashboard, DashboardSection, DashboardWidget components
 */
public class DashboardComponentHandle extends CustomComponentHandle {
    private static final String DASHBOARD_TAG = "Dashboard";
    private static final String DASHBOARD_SECTION_TAG = "DashboardSection";
    private static final String DASHBOARD_WIDGET_TAG = "DashboardWidget";

    private static final String SECTION_TITLE_PROP_KEY = "sectionTitle";
    private static final String WIDGET_TITLE_PROP_KEY = "widgetTitle";
    private static final String STYLE = "style";

    @Override
    public List<VariableDeclarator> createComponentStatements(JavaRewriter javaRewriter, JavaComponent component,
            InsertionPoint insertionPoint, JavaComponent parent, String layoutVariableName,
            ComponentInfo referenceComponent, JavaRewriter.AddTemplateOptions options) {
        String componentClassName = FlowComponentQuirks.getClassForComponent(component);
        ClassOrInterfaceType fullType = StaticJavaParser.parseClassOrInterfaceType(componentClassName); // Import
        CompilationUnit compilationUnit = insertionPoint.getCompilationUnit();

        JavaRewriterUtil.addImport(compilationUnit, fullType.getNameWithScope());
        String variableName = JavaRewriterUtil.generateVariableName(component, fullType, insertionPoint);
        VariableDeclarator variableDeclarator = new VariableDeclarator(fullType.removeScope(), variableName);

        ObjectCreationExpr creationExpr = new ObjectCreationExpr();
        creationExpr.setType(fullType.removeScope());
        variableDeclarator.setInitializer(creationExpr);

        VariableDeclarationExpr varDeclaration = new VariableDeclarationExpr(variableDeclarator);

        insertionPoint.add(new ExpressionStmt(varDeclaration));
        if (DASHBOARD_SECTION_TAG.equals(component.tag()) && component.props().containsKey(SECTION_TITLE_PROP_KEY)) {
            creationExpr.addArgument(new StringLiteralExpr((String) component.props().get(SECTION_TITLE_PROP_KEY)));
            component.props().remove(SECTION_TITLE_PROP_KEY);
        }
        if (DASHBOARD_WIDGET_TAG.equals(component.tag()) && component.props().containsKey(WIDGET_TITLE_PROP_KEY)) {
            creationExpr.addArgument(new StringLiteralExpr((String) component.props().get(WIDGET_TITLE_PROP_KEY)));
            component.props().remove(WIDGET_TITLE_PROP_KEY);
        }
        replaceCssVariablesWithMethodCalls(component, insertionPoint, variableDeclarator.getNameAsExpression());
        // Properties setters
        for (Map.Entry<String, Object> entry : component.props().entrySet()) {
            String prop = entry.getKey();
            Object value = entry.getValue();
            // skip empty styles
            if (STYLE.equals(prop) && value instanceof Map<?, ?> styleProperties && styleProperties.isEmpty()) {
                continue;
            }
            JavaRewriter.SetterAndValue setterAndValue = JavaRewriterUtil
                    .getSetterAndValue(JavaRewriterUtil.getClass(componentClassName), prop, value);
            javaRewriter.insertSetter(insertionPoint, new NameExpr(variableName), setterAndValue.setter(),
                    setterAndValue.value(), component);
        }

        if (!component.children().isEmpty()) {
            javaRewriter.createComponentStatements(insertionPoint, component, component.children(), variableName, null,
                    options, null);
        }
        javaRewriter.attachComponent(insertionPoint, component, parent, layoutVariableName, referenceComponent,
                new NameExpr(variableName), variableName, options, null);

        return List.of(variableDeclarator);
    }

    private void replaceCssVariablesWithMethodCalls(JavaComponent component, InsertionPoint insertionPoint,
            NameExpr scope) {
        if ((DASHBOARD_TAG.equals(component.tag()))) {
            Optional<Map<?, ?>> mapStylePropertiesOptional = getMapStyleProperties(component);
            if (mapStylePropertiesOptional.isEmpty()) {
                return;
            }
            Map<?, ?> styleProperties = mapStylePropertiesOptional.get();
            styleProperties.forEach((key, value) -> {
                if (key.equals("--vaadin-dashboard-col-min-width")) {
                    MethodCallExpr expr = new MethodCallExpr("setMinimumColumnWidth",
                            new StringLiteralExpr(value.toString()));
                    expr.setScope(scope);
                    insertionPoint.add(new ExpressionStmt(expr));
                } else if (key.equals("--vaadin-dashboard-col-max-count")) {
                    MethodCallExpr expr = new MethodCallExpr("setMaximumColumnCount",
                            new IntegerLiteralExpr(value.toString()));
                    expr.setScope(scope);
                    insertionPoint.add(new ExpressionStmt(expr));
                }
            });
            styleProperties.remove("--vaadin-dashboard-col-min-width");
            styleProperties.remove("--vaadin-dashboard-col-max-count");
            return;
        }
        if (DASHBOARD_WIDGET_TAG.equals(component.tag())) {
            Optional<Map<?, ?>> mapStylePropertiesOptional = getMapStyleProperties(component);
            if (mapStylePropertiesOptional.isEmpty()) {
                return;
            }
            Map<?, ?> styleProperties = mapStylePropertiesOptional.get();
            styleProperties.forEach((key, value) -> {
                if (key.equals("--vaadin-dashboard-item-rowspan")) {
                    MethodCallExpr expr = new MethodCallExpr("setRowspan", new IntegerLiteralExpr(value.toString()));
                    expr.setScope(scope);
                    insertionPoint.add(new ExpressionStmt(expr));
                } else if (key.equals("--vaadin-dashboard-item-colspan")) {
                    MethodCallExpr expr = new MethodCallExpr("setColspan", new IntegerLiteralExpr(value.toString()));
                    expr.setScope(scope);
                    insertionPoint.add(new ExpressionStmt(expr));
                }
            });
            styleProperties.remove("--vaadin-dashboard-item-rowspan");
            styleProperties.remove("--vaadin-dashboard-item-colspan");
        }
    }

    private Optional<Map<?, ?>> getMapStyleProperties(JavaComponent component) {
        if (!component.props().containsKey(STYLE)) {
            return Optional.empty();
        }
        if (component.props().get(STYLE) instanceof Map<?, ?> styleProperties) {
            return Optional.of(styleProperties);
        }
        return Optional.empty();
    }
}
