package com.vaadin.copilot.ide;

import java.io.File;
import java.io.IOException;
import java.util.Optional;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.server.VaadinSession;

import elemental.json.JsonObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpenComponentInIDE extends CopilotCommand {

    IdeIntegration ideIntegration;

    CopilotIDEPlugin idePlugin;

    public OpenComponentInIDE() {
        this.ideIntegration = new IdeIntegration(getVaadinSession());
        this.idePlugin = CopilotIDEPlugin.getInstance();
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("show-in-ide")) {
            if (data.hasKey("nodeId")) {
                showFlowComponent((int) data.getNumber("uiId"), (int) data.getNumber("nodeId"),
                        data.hasKey("attach") && data.getBoolean("attach"));
            } else if (data.hasKey("javaClassName")) {
                String javaClassName = data.getString("javaClassName");
                showJavaClass(javaClassName);
            } else {
                showFile(data.getString("fileName"), (int) data.getNumber("lineNumber"));
            }
            return true;
        }
        return false;
    }

    public void showFlowComponent(int uiId, int nodeId, boolean attach) {
        VaadinSession session = getVaadinSession();
        session.access(() -> {
            Element element = session.findElement(uiId, nodeId);
            Optional<Component> c = element.getComponent();
            if (c.isPresent()) {
                if (attach) {
                    ideIntegration.showComponentAttachInIde(c.get());
                } else {
                    ideIntegration.showComponentCreateInIde(c.get());
                }

            } else {
                getLogger().error(
                        "Only component locations are tracked. The given node id refers to an element and not a component");
            }
        });
    }

    public void showFile(String filename, int lineNumber) {
        File file = new File(filename);
        if (!file.toPath().isAbsolute()) {
            try {
                file = new File(getProjectFileManager().makeAbsolute(filename));
            } catch (IOException e) {
                getLogger().warn("File not found: " + filename);
                return;
            }
        }
        IdeUtils.openFile(file, lineNumber);
    }

    private void showJavaClass(String javaClassName) {
        File fileForClass = getProjectFileManager().getFileForClass(javaClassName);
        if (fileForClass != null && fileForClass.exists()) {
            showFile(fileForClass.getAbsolutePath(), 1);
        } else {
            getLogger().warn("File not found for {} class ", javaClassName);
        }
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
