package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.util.Optional;

import com.vaadin.copilot.customcomponent.CustomComponent;
import com.vaadin.flow.component.Component;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;

public record ComponentInfo(Class<? extends Component> type, ObjectCreationExpr objectCreationExpr,
        BlockStmt componentCreateScope, VariableDeclarator localVariableDeclarator, AssignExpr assignmentExpression,
        FieldDeclaration fieldDeclaration, FieldDeclaration fieldDeclarationAndAssignment, String localVariableName,
        String fieldName, ConstructorDeclaration routeConstructor, boolean containerComposite,
        boolean isAnonymousComponent, boolean isReturnValue, Optional<CustomComponent> customComponentInfo,
        Optional<ComponentAttachInfo> componentAttachInfoOptional,
        Optional<ComponentCreateInfo> componentCreateInfoOptional) {

    public AttachExpression getAttachCallInSameFileOrThrow() {
        if (componentAttachInfoOptional.isEmpty()) {
            return null;
        }
        ComponentAttachInfo componentAttachInfo = componentAttachInfoOptional.get();
        if (!componentAttachInfo.inSameFileWithCreateLocation()) {
            throw new IllegalArgumentException(
                    "Create and attach location are in different files. This operation is not supported at the moment");
        }
        return componentAttachInfo.attachCall();
    }

    public Optional<BlockStmt> getComponentAttachScopeOrThrowIfDifferentFile() {
        if (componentAttachInfoOptional.isEmpty()) {
            return Optional.empty();
        }
        ComponentAttachInfo componentAttachInfo = componentAttachInfoOptional.get();
        if (!componentAttachInfo.inSameFileWithCreateLocation()) {
            throw new IllegalArgumentException(
                    "Create and attach location are in different files. This operation is not supported at the moment");
        }
        return componentAttachInfo.componentAttachScope();
    }

    public File getCreateLocationFileOrThrowIfNull() {
        return componentCreateInfoOptional
                .orElseThrow(() -> new IllegalArgumentException("Create location rewriter is null")).file();
    }

    public CompilationUnit getCreateLocationCompilationUnitOrThrowIfNull() {
        return componentCreateInfoOptional
                .orElseThrow(() -> new IllegalArgumentException("Create location compilation unit is null!"))
                .javaSource().getCompilationUnit();
    }

    public CompilationUnit getAttachLocationCompilationUnitOrThrowIfNull() {
        return componentAttachInfoOptional
                .orElseThrow(() -> new IllegalArgumentException("Attach location compilation unit is null!"))
                .javaSource().getCompilationUnit();
    }

    /**
     * Gets the variable declarator of the component, either for the local variable
     * or field.
     *
     * @return the variable declarator of the component, or null if not found
     */
    public VariableDeclarator getVariableDeclarator() {
        if (localVariableDeclarator() != null) {
            return localVariableDeclarator();
        } else if (fieldDeclaration() != null) {
            return fieldDeclaration().getVariables().stream()
                    .filter(variable -> variable.getNameAsString().equals(fieldName())).findFirst().orElse(null);
        }

        return null;
    }

    public String typeWithGenerics() {
        VariableDeclarator variableDeclarator = getVariableDeclarator();
        String typeFromClass = type().getSimpleName();
        if (variableDeclarator == null) {
            return typeFromClass;
        }
        if (!variableDeclarator.getType().isVarType()) {
            return variableDeclarator.getType().toString();
        }
        return variableDeclarator.getInitializer().filter(Expression::isObjectCreationExpr)
                .map(ObjectCreationExpr.class::cast).map(a -> a.getType().toString()).orElse(typeFromClass);
    }
}
