package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.vaadin.copilot.CopilotException;
import com.vaadin.copilot.KotlinNotSupportedException;
import com.vaadin.copilot.KotlinUtil;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.ide.CopilotIDEPlugin;

/**
 * Provider that makes sure same JavaRewriter is served for the same file so
 * same file should not be overwritten in the same operation.
 */
public class JavaFileSourceProvider {
    public static final String UNDO_LABEL = CopilotIDEPlugin.undoLabel("Java view update");
    private final Map<String, JavaSource> filePathJavaFileMap = new HashMap<>();

    /**
     * Retrieves a {@link JavaSource} representation of the specified Java source
     * file.
     * <p>
     * If the source has already been loaded previously, the cached
     * {@code JavaSource} is returned. Otherwise, the file is read from disk,
     * validated to ensure it is not a Kotlin file, and then parsed into a new
     * {@code JavaSource} instance which is cached for future use.
     *
     * @param sourceFile
     *            the Java source file to load
     * @return the {@code JavaSource} representation of the given file
     * @throws IOException
     *             if an I/O error occurs while reading the file
     * @throws KotlinNotSupportedException
     *             if the file is determined to be a Kotlin file
     */
    public JavaSource getJavaSource(File sourceFile) throws IOException {
        if (filePathJavaFileMap.containsKey(sourceFile.getAbsolutePath())) {
            return filePathJavaFileMap.get(sourceFile.getAbsolutePath());
        }
        KotlinUtil.throwIfKotlin(sourceFile);
        String fileContent = ProjectFileManager.get().readFile(sourceFile);
        JavaSource javaSource = new JavaSource(fileContent);
        filePathJavaFileMap.put(sourceFile.getAbsolutePath(), javaSource);
        return javaSource;
    }

    public void saveComponentInfoSourceFiles(ComponentInfo componentInfo) {
        componentInfo.componentCreateInfoOptional()
                .ifPresent(componentCreateInfo -> saveJavaSourceIfChanged(componentCreateInfo.javaSource(),
                        componentCreateInfo.file()));
        componentInfo.componentAttachInfoOptional()
                .ifPresent(componentAttachInfo -> saveJavaSourceIfChanged(componentAttachInfo.javaSource(),
                        componentAttachInfo.file()));
    }

    private void saveJavaSourceIfChanged(JavaSource javaSource, File file) {
        if (javaSource.isChanged() && !javaSource.isWritten()) {
            String result = javaSource.getResult();
            try {
                ProjectFileManager.get().writeFile(file, UNDO_LABEL, result);
            } catch (IOException e) {
                throw new CopilotException("Unable to save Java source", e);
            }
            javaSource.setWritten(true);
        }
    }

    /**
     * Clears all cached {@link JavaSource} objects from memory.
     * <p>
     * Note: This method should be called anytime a new Java related operation
     * begins.
     * </p>
     */
    public void clear() {
        filePathJavaFileMap.clear();
    }
}
