package com.vaadin.copilot;

import java.io.IOException;
import java.net.ConnectException;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.server.frontend.ProxyFactory;
import com.vaadin.flow.server.frontend.installer.ProxyConfig;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;
import com.vaadin.pro.licensechecker.ProKey;

import elemental.json.JsonObject;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import io.netty.buffer.ByteBufAllocator;
import io.netty.handler.codec.http.HttpMethod;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Disposable;
import reactor.core.publisher.Mono;
import reactor.netty.ByteBufMono;

/**
 * Base class for Copilot Server API clients
 */
public class CopilotServerClient {

    protected static final String SERVER_URL_ENV = "copilot.serverBaseUrl";
    private static final String SERVER_PORT_ENV = "COPILOT_SERVER_PORT";
    private static String serverBaseUrlCache = null;

    private HttpClient _httpClient;

    private final ObjectMapper objectMapper;

    /**
     * Constructor initializing HttpClient and ObjectMapper
     */
    public CopilotServerClient() {
        this.objectMapper = new ObjectMapper();
        this.objectMapper.registerModule(new JavaTimeModule());
        this.objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    public HttpClient getHttpClient() {
        if (_httpClient == null) {
            List<ProxyConfig.Proxy> proxies = ProxyFactory.getProxies(ProjectFileManager.get().getProjectRoot());
            ProxyConfig.Proxy proxy = new ProxyConfig(proxies).getProxyForUrl(getServerBaseUrl());
            ProxySelector proxySelector;
            if (proxy != null) {
                proxySelector = ProxySelector.of(new InetSocketAddress(proxy.host, proxy.port));
            } else {
                proxySelector = ProxySelector.getDefault();
            }
            HttpClient.Builder httpClientBuilder = HttpClient.newBuilder().version(HttpClient.Version.HTTP_2)
                    .followRedirects(HttpClient.Redirect.NORMAL).proxy(proxySelector);
            _httpClient = httpClientBuilder.build();
        }
        return _httpClient;
    }

    public HttpRequest buildRequest(URI uri, String json) {
        return HttpRequest.newBuilder().uri(uri).POST(HttpRequest.BodyPublishers.ofString(json))
                .header("Content-Type", "application/json; charset=utf-8").timeout(Duration.ofSeconds(120)).build();
    }

    public String writeAsJsonString(Object obj) {
        try {
            return objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    public <T> T readValue(String string, Class<T> clazz) {
        try {
            return objectMapper.readValue(string, clazz);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    public URI getQueryURI(String path) {
        try {
            return new URI(getServerBaseUrl() + path);
        } catch (URISyntaxException e) {
            throw new IllegalStateException("Invalid server configuration, server uri is wrong", e);
        }
    }

    protected static String getServerBaseUrl() {
        if (serverBaseUrlCache != null) {
            return serverBaseUrlCache;
        }
        serverBaseUrlCache = System.getenv(SERVER_URL_ENV);
        if (serverBaseUrlCache != null) {
            return serverBaseUrlCache;
        }

        if (Copilot.isDevelopmentMode()) {
            // Try a localhost server
            Integer serverPort = Optional.ofNullable(System.getenv(SERVER_PORT_ENV)).map(Integer::parseInt)
                    .orElse(8081);
            String localhostUrl = "http://localhost:%s/v1/".formatted(serverPort);

            // Check if the local server is running
            try {
                URL statusUrl = new URL(localhostUrl + "actuator/health");
                String status = IOUtils.toString(statusUrl, StandardCharsets.UTF_8);
                if (status.equals("{\"status\":\"UP\"}")) {
                    getLogger().info("Using the local server for development.");
                    serverBaseUrlCache = localhostUrl;
                    return serverBaseUrlCache;
                }
                getLogger().warn("No local server found. Something else running on {}?", statusUrl);
            } catch (ConnectException e) {
                // It's just not running - that's fine
                getLogger().debug(
                        "No local server found, as this exception occurred while trying to reach the local server", e);
            } catch (IOException e) {
                getLogger().warn(
                        "No local server found, as this exception occurred while trying to reach the local server", e);
            }
            getLogger().info("Using the staging server for development.");
            serverBaseUrlCache = "https://copilot.stg.vaadin.com/v1/";
            return serverBaseUrlCache;
        } else {
            serverBaseUrlCache = "https://copilot.vaadin.com/v1/";
            return serverBaseUrlCache;
        }
    }

    public <REQ, RESP> void sendCopilotRequest(String path, REQ request, Class<RESP> responseType,
            Consumer<RESP> responseHandler, DevToolsInterface devToolsInterface, String command, JsonObject respData) {
        URI uri = getQueryURI(path);
        HttpRequest httpRequest = buildRequest(uri, writeAsJsonString(request));
        CompletableFuture<HttpResponse<String>> resp = getHttpClient().sendAsync(httpRequest,
                HttpResponse.BodyHandlers.ofString());

        resp.whenComplete((httpResponse, throwable) -> {
            if (throwable != null || httpResponse.statusCode() != 200) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, "Internal error in service",
                        throwable);
                return;
            }
            RESP response = readValue(httpResponse.body(), responseType);
            responseHandler.accept(response);
        }).whenComplete((httpResponse, throwable) -> {
            if (throwable != null) {
                // This is an exception thrown in the response handler
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData,
                        "Internal error in response handler", throwable);
            }
        });

    }

    public Disposable sendReactive(String path, String json) {
        return reactor.netty.http.client.HttpClient.create()
                .headers(headers -> headers.set("Content-Type", "application/json; charset=utf-8"))
                .request(HttpMethod.POST)
                .send(ByteBufMono.fromString(Mono.just(json), StandardCharsets.UTF_8, ByteBufAllocator.DEFAULT))
                .uri(getQueryURI(path)).response().subscribe();
    }

    public record AccessControlData(String proKey, String machineId, String copilotVersion) {
        public static AccessControlData create() {
            ProKey proKey = LocalProKey.get();

            return new AccessControlData(proKey == null ? null : proKey.getProKey(), MachineId.get(),
                    CopilotVersion.getVersion());
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(CopilotServerClient.class);
    }
}
