package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Optional;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ide.CopilotIDEPlugin;

import elemental.json.Json;
import elemental.json.JsonObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProjectFileHandler extends CopilotCommand {

    private static final String COPILOT_UNDO_LABEL = CopilotIDEPlugin.undoLabel("File Update");
    private static final Logger log = LoggerFactory.getLogger(ProjectFileHandler.class);

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-project-file")) {
            String filename = data.getString("filename");
            String reqId = data.getString(KEY_REQ_ID);

            JsonObject responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);
            try {
                responseData.put("content", getProjectFileManager().readFile(filename));
                devToolsInterface.send(Copilot.PREFIX + "project-file", responseData);
            } catch (IOException e) {
                getLogger().debug("Unable to read project file {}", filename, e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData, "Unable to read file", e);
            }
            return true;
        } else if (command.equals("write-project-file")) {
            String filename = data.getString("filename");
            String reqId = data.getString(KEY_REQ_ID);
            String content = data.getString("content");
            JsonObject responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);
            try {
                getProjectFileManager().writeFile(filename, COPILOT_UNDO_LABEL, content);
                devToolsInterface.send(Copilot.PREFIX + "project-file-written", responseData);
            } catch (IOException e) {
                getLogger().debug("Unable to write project file {}", filename, e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData, "Unable to write file", e);
            }
            return true;
        } else if (command.equals("write-project-resource-binary")) {
            JsonObject responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));

            Optional<Path> resourceFolder = Util.findCurrentModule(getVaadinSession(), data.getObject("currentView"))
                    .flatMap(JavaSourcePathDetector.ModuleInfo::getOrGuessResourceFolder);
            String filename = data.getString("filename");
            String base64 = data.getString("base64Content");
            boolean renameIfExists = data.getBoolean("renameIfExists");
            try {
                if (resourceFolder.isEmpty()) {
                    throw new IOException("Unable to determine resource folder for the current view");
                }
                File written = getProjectFileManager().writeFileBase64(
                        resourceFolder.get().resolve(filename).toString(), COPILOT_UNDO_LABEL, base64, renameIfExists);
                responseData.put("filename", written.getName());
                devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            } catch (IOException e) {
                responseData.put("error", true);
                devToolsInterface.send(Copilot.PREFIX + "project-file-written", responseData);
                getLogger().error("Unable to write project file {}", filename, e);
            }
            return true;
        }

        return false;
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
