package com.vaadin.copilot.ai;

import java.util.List;
import java.util.Map;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.copilot.UIServiceCreator;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.flow.internal.JsonUtils;

import elemental.json.Json;
import elemental.json.JsonObject;

public class AICreateFieldsHandler extends CopilotCommand {

    private final CopilotServerClient copilotServerClient = new CopilotServerClient();

    public record CreateBeanFieldsData(String name, String description) {

    }

    public record CreateBeanFieldsRequest(CreateBeanFieldsData data,
            CopilotServerClient.AccessControlData accessControlData) {
    }

    public record CreateBeanFieldsResponse(List<UIServiceCreator.FieldInfo> fieldInfos) {

    }

    public record CreateExampleDataData(String entityName, List<UIServiceCreator.FieldInfo> fieldInfos, int count) {
    }

    public record CreateExampleDataRequest(CreateExampleDataData data,
            CopilotServerClient.AccessControlData accessControlData) {
    }

    public record CreateExampleDataResponse(List<Map<String, String>> exampleData) {
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("ai-create-bean-fields")) {
            UserInfoServerClient.throwIfAIUsageDisabled();
            JsonObject respData = Json.createObject();
            respData.put(CopilotCommand.KEY_REQ_ID, data.getString(CopilotCommand.KEY_REQ_ID));

            CreateBeanFieldsRequest request = new CreateBeanFieldsRequest(
                    new CreateBeanFieldsData(data.getString("name"), data.getString("description")),
                    CopilotServerClient.AccessControlData.create());

            copilotServerClient.sendCopilotRequest("create-bean-fields", request, CreateBeanFieldsResponse.class,
                    (response) -> {
                        respData.put("fieldInfos", JsonUtils.listToJson(response.fieldInfos()));

                        devToolsInterface.send(command + "-resp", respData);
                    }, devToolsInterface, command, respData);

            return true;
        } else if (command.equals("ai-generate-example-data")) {
            JsonObject respData = Json.createObject();
            respData.put(CopilotCommand.KEY_REQ_ID, data.getString(CopilotCommand.KEY_REQ_ID));
            List<UIServiceCreator.FieldInfo> fieldInfos = JsonUtils.stream(data.getArray("fieldInfos"))
                    .map(JsonObject.class::cast).map(UIServiceCreator.FieldInfo::fromJson).toList();
            CreateExampleDataData exampleDataData = new CreateExampleDataData(data.getString("entityName"), fieldInfos,
                    10);
            copilotServerClient.sendCopilotRequest("create-example-data",
                    new CreateExampleDataRequest(exampleDataData, CopilotServerClient.AccessControlData.create()),
                    CreateExampleDataResponse.class, exampleDataResponse -> {
                        List<Map<String, String>> exampleData = exampleDataResponse.exampleData();
                        respData.put("exampleData", JsonUtils.listToJson(exampleData));
                        devToolsInterface.send(command + "-resp", respData);
                    }, devToolsInterface, command, respData);
            return true;
        }
        return false;

    }

}
