package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.vaadin.copilot.ComponentSourceFinder;
import com.vaadin.copilot.CopilotException;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.Composite;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.server.VaadinSession;

import com.github.javaparser.Position;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;

/**
 * Static source file parser for the given component class. See
 * {@link #findCreateLocationCandidateOrThrow(VaadinSession)}
 */
public class ClassSourceStaticAnalyzer {
    private static final String INIT_METHOD_NAME = "<init>";
    private final ComponentSourceFinder sourceFinder;
    private final Component component;
    private final File file;

    /**
     * Constructs a {@link ClassSourceStaticAnalyzer}
     *
     * @param sourceFinder
     *            to get children of the given component if necessary
     * @param component
     *            Component instance that is created with
     * @param file
     *            Source file of the component
     */
    public ClassSourceStaticAnalyzer(ComponentSourceFinder sourceFinder, Component component, File file) {
        this.sourceFinder = sourceFinder;
        this.file = file;
        this.component = component;
    }

    /**
     * Returns the class declaration statement line number of the given class. This
     * method should be called when candidate constructor is not found
     *
     * @param classOrInterfaceDeclaration
     *            Class
     * @return Beginning of the class declaration statement, throws {@link }
     */
    private Optional<MethodAndLineNumber> findClassDeclarationBeginLine(
            Set<MethodAndLineNumber> candidateMethodAndLineNumbers,
            ClassOrInterfaceDeclaration classOrInterfaceDeclaration) {
        if (!candidateMethodAndLineNumbers.isEmpty()) {
            return Optional.empty();
        }
        Optional<Position> position = classOrInterfaceDeclaration.getBegin();
        return position.map(value -> new MethodAndLineNumber(INIT_METHOD_NAME, value.line));

    }

    /**
     * Determines the candidate create location. There are three ways to find the
     * candidate, it is done in the following order:
     * <p>
     * <ol>
     * <li>Using the reference of children create locations to find candidate
     * constructor</li>
     * <li>Returning default constructor if present</li>
     * <li>Returning class declaration as fallback</li>
     * </ol>
     * </p>
     * Throws an exception if no candidate found from approaches above
     *
     * @return Create location of the given source file
     * @throws IOException
     *             thrown if file operation fails
     */
    public ComponentTracker.Location findCreateLocationCandidateOrThrow(VaadinSession vaadinSession)
            throws IOException {

        CompilationUnit compilationUnit = new JavaSource(ProjectFileManager.get().readFile(this.file))
                .getCompilationUnit();
        ClassOrInterfaceDeclaration classOrInterfaceDeclaration = findRelevantClassDeclaration(compilationUnit,
                component.getClass());
        Set<MethodAndLineNumber> candidateMethodAndLineNumbers = findChildrenAttachLocationLineNumbers(vaadinSession,
                component, file);
        Optional<MethodAndLineNumber> candidateConstructor = findCandidateConstructorWithChildrenAttachReferences(
                candidateMethodAndLineNumbers, compilationUnit, classOrInterfaceDeclaration.getNameAsString());
        if (candidateConstructor.isPresent()) {
            return createLocation(candidateConstructor.get());
        }
        Optional<MethodAndLineNumber> candidateDefaultConstructor = findDefaultConstructorCandidate(
                classOrInterfaceDeclaration);
        if (candidateDefaultConstructor.isPresent()) {
            return createLocation(candidateDefaultConstructor.get());
        }
        Optional<MethodAndLineNumber> candidateClassDeclaration = findClassDeclarationBeginLine(
                candidateMethodAndLineNumbers, classOrInterfaceDeclaration);
        if (candidateClassDeclaration.isPresent()) {
            return createLocation(candidateClassDeclaration.get());
        }
        throw new CopilotException("Could not find candidate constructor or method to modify");
    }

    private ClassOrInterfaceDeclaration findRelevantClassDeclaration(CompilationUnit compilationUnit, Class<?> clazz) {
        ClassOrInterfaceDeclaration topMostClassDeclaration = compilationUnit.getTypes().stream()
                .filter(ClassOrInterfaceDeclaration.class::isInstance).map(ClassOrInterfaceDeclaration.class::cast)
                .findFirst().orElseThrow(() -> new CopilotException("Could not find class or instantiation for class"));

        if (!clazz.getName().contains("$")) {
            return topMostClassDeclaration;
        }
        ClassOrInterfaceDeclaration targetClassToFindInnerClass = topMostClassDeclaration;
        String name = clazz.getName();
        String simpleClassNames = name.substring(name.indexOf('$') + 1);
        String[] innerClassNamesInOrder = simpleClassNames.split("\\$");

        for (String innerClassName : innerClassNamesInOrder) {
            targetClassToFindInnerClass = targetClassToFindInnerClass.getChildNodes().stream()
                    .filter(ClassOrInterfaceDeclaration.class::isInstance).map(ClassOrInterfaceDeclaration.class::cast)
                    .filter(innerClassDeclaration -> innerClassDeclaration.getNameAsString().equals(innerClassName))
                    .findFirst().orElseThrow(
                            () -> new CopilotException("Could not find inner class with name = " + innerClassName));
        }
        return targetClassToFindInnerClass;
    }

    private ComponentTracker.Location createLocation(MethodAndLineNumber methodAndLineNumber) {
        return new ComponentTracker.Location(this.component.getClass().getName(), file.getName(),
                methodAndLineNumber.methodName(), methodAndLineNumber.lineNumber());
    }

    private Optional<MethodAndLineNumber> findDefaultConstructorCandidate(
            ClassOrInterfaceDeclaration classOrInterfaceDeclaration) {
        return classOrInterfaceDeclaration.getDefaultConstructor().flatMap(ConstructorDeclaration::getBegin)
                .map(pos -> new MethodAndLineNumber(INIT_METHOD_NAME, pos.line));
    }

    /**
     * Finds the candidate constructor where components are added in the given class
     *
     * @param candidateMethodAndLineNumbers
     *            Line number and method name pairs of children, provided in runtime
     *            by Flow
     * @param compilationUnit
     *            Compilation unit of the given source file
     * @param className
     *            Class name
     * @return the line number if found, throws {@link CopilotException} with a
     *         meaningful message about failure
     */
    private Optional<MethodAndLineNumber> findCandidateConstructorWithChildrenAttachReferences(
            Set<MethodAndLineNumber> candidateMethodAndLineNumbers, CompilationUnit compilationUnit, String className) {
        ConstructorDeclaration candidateConstructorDeclaration = null;
        for (MethodAndLineNumber candidate : candidateMethodAndLineNumbers) {
            if (!INIT_METHOD_NAME.equals(candidate.methodName)) {
                continue;
            }
            Optional<ConstructorDeclaration> nodeOfType = JavaRewriterUtil.findNodeOfType(compilationUnit,
                    candidate.lineNumber, ConstructorDeclaration.class);
            if (nodeOfType.isPresent()) {
                ConstructorDeclaration constructorDeclaration = nodeOfType.get();
                if (candidateConstructorDeclaration == null) {
                    candidateConstructorDeclaration = constructorDeclaration;
                } else if (!candidateConstructorDeclaration.equals(constructorDeclaration)) {
                    throw new CopilotException("Multiple candidate constructors found for " + className);
                }
            }
        }

        if (candidateConstructorDeclaration == null) {
            return Optional.empty();
        }
        Optional<Position> beginPosition = candidateConstructorDeclaration.getBegin();
        if (beginPosition.isEmpty()) {
            throw new CopilotException("Could not find the line number of the candidate constructor for " + className);
        }
        return Optional.of(new MethodAndLineNumber(INIT_METHOD_NAME, beginPosition.get().line));
    }

    private Set<MethodAndLineNumber> findChildrenAttachLocationLineNumbers(VaadinSession vaadinSession,
            Component component, File file) {
        String className = component.getClass().getName();
        Set<MethodAndLineNumber> childrenAttachLocationLineNumbers = new HashSet<>();
        vaadinSession.accessSynchronously(() -> {
            boolean compositeComponent = component instanceof Composite<?>;
            List<Component> candidateChildren = component.getChildren().toList();

            if (compositeComponent && candidateChildren.size() == 1) {
                // first child is the provided component using reflection by Flow if given
                // component inherits Composite.
                candidateChildren = ((Composite<?>) component).getContent().getChildren().toList();
            }

            candidateChildren.forEach(child -> {
                ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder._getSourceLocation(child);
                if (componentTypeAndSourceLocation.attachLocationInProject().isPresent()) {
                    ComponentTracker.Location attachLocation = componentTypeAndSourceLocation
                            .getAttachLocationOrThrow();
                    if (attachLocation.filename().equals(file.getName())
                            && attachLocation.className().equals(className)) {
                        childrenAttachLocationLineNumbers
                                .add(new MethodAndLineNumber(attachLocation.methodName(), attachLocation.lineNumber()));
                    }
                }
            });
        });
        return childrenAttachLocationLineNumbers;
    }

    /**
     * Record that contains method name and line number
     *
     * @param methodName
     *            Method name, for constructor it is {@link #INIT_METHOD_NAME}
     * @param lineNumber
     *            line number of beginning position of method
     */
    private record MethodAndLineNumber(String methodName, Integer lineNumber) {
        @Override
        public boolean equals(Object o) {
            if (o == null || getClass() != o.getClass())
                return false;
            MethodAndLineNumber that = (MethodAndLineNumber) o;
            return Objects.equals(methodName, that.methodName) && Objects.equals(lineNumber, that.lineNumber);
        }

        @Override
        public int hashCode() {
            return Objects.hash(methodName, lineNumber);
        }
    }

}
