package com.vaadin.copilot.plugins.docs;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.CopilotServerClient;

import elemental.json.Json;
import elemental.json.JsonObject;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DocsHandler extends CopilotCommand {

    private final ObjectMapper objectMapper;
    private final DocsDataLoader docsDataLoader;
    private final CopilotServerClient copilotServerClient = new CopilotServerClient();

    public record AskDocsRequest(AskDocsData data, CopilotServerClient.AccessControlData accessControlData) {

    }

    public record AskDocsData(String question, String framework) {

    }

    public record AskDocsResponse(String answer, String error) {
    }

    public DocsHandler() {
        this.objectMapper = new ObjectMapper();
        this.docsDataLoader = new DocsDataLoader();
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-docs")) {
            var reqId = data.getString(KEY_REQ_ID);
            var responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);
            try {
                DocsRequest docsRequest = objectMapper.readValue(data.toJson(), DocsRequest.class);
                DocsResponse response = docsDataLoader.get(docsRequest);
                responseData.put("content", objectMapper.writeValueAsString(response));
            } catch (DocsException exception) {
                responseData.put("errorCode", exception.getCode());
            } catch (Exception ex) {
                responseData.put("errorCode", DocsException.UNKNOWN_ERROR);
                getLogger().error(ex.getMessage(), ex);
            }
            devToolsInterface.send("copilot-get-docs-response", responseData);

            return true;
        } else if (command.equals("ask-docs")) {
            var reqId = data.getString(KEY_REQ_ID);
            var responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);
            AskDocsData askData = new AskDocsData(data.getString("question"), data.getString("framework"));
            copilotServerClient.sendCopilotRequest("ask-docs",
                    new AskDocsRequest(askData, CopilotServerClient.AccessControlData.create()), AskDocsResponse.class,
                    response -> {
                        responseData.put("answer", response.answer() != null ? response.answer() : response.error());
                        devToolsInterface.send(command + "-response", responseData);
                    }, devToolsInterface, command, responseData);
            return true;
        }
        return false;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(DocsHandler.class);
    }

    @Override
    public boolean canBeParallelCommand(String command) {
        return command.equalsIgnoreCase("get-docs");
    }
}
