package com.vaadin.copilot.plugins.vaadinversionupdate;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Predicate;

import com.vaadin.copilot.CopilotException;
import com.vaadin.copilot.PomFileRewriter;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

/**
 * Handles updating the Vaadin version in a project using Maven.
 */
public class VaadinVersionUpdate {
    private static final String VAADIN_PRE_RELEASES_ID = "vaadin-prereleases";
    private static final String VAADIN_PRE_RELEASES_URL = "https://maven.vaadin.com/vaadin-prereleases/";
    private final ApplicationConfiguration applicationConfiguration;

    /**
     * Constructs a {@code VaadinVersionUpdate} instance.
     *
     * @param applicationConfiguration
     *            the application configuration providing project settings
     */
    public VaadinVersionUpdate(ApplicationConfiguration applicationConfiguration) {
        this.applicationConfiguration = applicationConfiguration;
    }

    /**
     * Updates the Vaadin version in the project's configuration files.
     *
     * @param newVersion
     *            the new Vaadin version to set
     * @param preRelease
     *            <code>true</code> if version is preRelease, <code>false</code>
     *            otherwise
     * @throws CopilotException
     *             if the Vaadin BOM dependency is not found or if Gradle is used
     */
    public void updateVaadinVersion(String newVersion, boolean preRelease) {
        File projectFolder = applicationConfiguration.getProjectFolder();
        File projectRoot = ProjectFileManager.get().getProjectRoot();
        Path pomFilePath = Path.of(projectFolder.getPath(), "pom.xml");

        if (!Files.exists(pomFilePath)) {
            Path gradleFilePath = Path.of(projectFolder.getPath(), "build.gradle");
            if (Files.exists(gradleFilePath)) {
                throw new CopilotException("Gradle is not supported yet.");
            } else {
                throw new CopilotException("Unable to detect build tool");
            }
        }

        Optional<PomFileRewriter.Dependency> vaadinBomDependencyInPomFile = findVaadinBomDependencyByWalkingPomFiles(
                projectFolder.toPath(), projectRoot.toPath());
        if (vaadinBomDependencyInPomFile.isEmpty()) {
            throw new CopilotException("Could not find vaadin-bom artifact in pom.xml files");
        }
        PomFileRewriter.Dependency dependency = vaadinBomDependencyInPomFile.get();

        if (dependency.version().startsWith("${") && dependency.version().endsWith("}")) {
            updateVaadinVersionPropertyInPomFileIfPresent(dependency.version(), newVersion, projectFolder.toPath(),
                    projectRoot.toPath());
            addPreReleaseRepositoryIfNeeded(preRelease, projectFolder.toPath(), projectRoot.toPath());
            addPluginRepositoryIfNeeded(preRelease, projectFolder.toPath(), projectRoot.toPath());
            return;
        }
        dependency.fileRewriter().updateDependencyVersion(dependency, newVersion);
        try {
            dependency.fileRewriter().save();
        } catch (IOException e) {
            throw new CopilotException(e);
        }
        addPreReleaseRepositoryIfNeeded(preRelease, projectFolder.toPath(), projectRoot.toPath());
        addPluginRepositoryIfNeeded(preRelease, projectFolder.toPath(), projectRoot.toPath());
    }

    private void addPreReleaseRepositoryIfNeeded(boolean preRelease, Path projectFolder, Path projectRoot) {
        if (!preRelease) {
            return;
        }

        try {
            PomFileRewriter foundFileRewriter = testPomFiles(projectFolder, projectRoot,
                    pomFileRewriter -> pomFileRewriter.getRepositoriesElement().isPresent());
            if (foundFileRewriter.hasRepositoryByUrl(VAADIN_PRE_RELEASES_URL)) {
                // Already there
                return;
            }
            if (foundFileRewriter.hasRepositoryById(VAADIN_PRE_RELEASES_ID)) {
                getLogger().warn("pom.xml already has a repository with id " + VAADIN_PRE_RELEASES_ID
                        + " but it does not refer to " + VAADIN_PRE_RELEASES_URL
                        + ". This might not work as expected.");
                return;
            }

            foundFileRewriter.addRepository(VAADIN_PRE_RELEASES_ID, VAADIN_PRE_RELEASES_URL);
            foundFileRewriter.save();
        } catch (Exception e) {
            throw new CopilotException("Could not update pom file", e);
        }
    }

    private void addPluginRepositoryIfNeeded(boolean preRelease, Path projectFolder, Path projectRoot) {
        if (!preRelease) {
            return;
        }
        try {
            PomFileRewriter foundFileRewriter = testPomFiles(projectFolder, projectRoot,
                    pomFileRewriter -> pomFileRewriter.getPluginRepositoriesElement().isPresent());
            foundFileRewriter.addPluginRepository(VAADIN_PRE_RELEASES_ID, VAADIN_PRE_RELEASES_URL);
            foundFileRewriter.save();
        } catch (Exception e) {
            throw new CopilotException("Could not update pom file", e);
        }
    }

    /**
     * Tests pom.xml files in the project starting from the projectFolder to
     * projectRoot to find pom.xml to update
     *
     * @param projectFolder
     *            Project folder of the project
     * @param projectRoot
     *            Project root of the project that might differ on multi module
     *            projects
     * @param predicate
     *            Predicate method to find PomFileRewriter
     * @return PomFileRewriter that will be updated
     * @throws IOException
     *             thrown if file operation fails
     * @throws SAXException
     *             thrown if xml parsing fails
     */
    private PomFileRewriter testPomFiles(Path projectFolder, Path projectRoot, Predicate<PomFileRewriter> predicate)
            throws IOException, SAXException {
        Path pomFileInProjectFolder = Path.of(projectFolder.toString(), "pom.xml");
        AtomicReference<PomFileRewriter> pomFileRewriterAtomicReference = new AtomicReference<>(
                new PomFileRewriter(pomFileInProjectFolder.toFile()));
        walkThroughPomFiles(projectFolder, projectRoot, pomFileRewriter -> {
            Boolean found = predicate.test(pomFileRewriter);
            if (Boolean.TRUE.equals(found)) {
                pomFileRewriterAtomicReference.set(pomFileRewriter);
                return true;
            }
            return false;
        });
        return pomFileRewriterAtomicReference.get();
    }

    /**
     * Searches for the Vaadin BOM dependency in the project's Maven configuration.
     *
     * @param pomContainingFolder
     *            the directory containing the pom.xml file
     * @param projectRootFolder
     *            the root directory of the project
     * @return an {@code Optional} containing the dependency if found, otherwise
     *         empty
     */
    private Optional<PomFileRewriter.Dependency> findVaadinBomDependencyByWalkingPomFiles(Path pomContainingFolder,
            Path projectRootFolder) {
        AtomicReference<Optional<PomFileRewriter.Dependency>> dependencyOptional = new AtomicReference<>(
                Optional.empty());
        walkThroughPomFiles(pomContainingFolder, projectRootFolder, model -> {
            PomFileRewriter.Dependency dependencyByGroupIdAndArtifactId = model
                    .findDependencyByGroupIdAndArtifactId("com.vaadin", "vaadin-bom");
            if (dependencyByGroupIdAndArtifactId != null) {
                dependencyOptional.set(Optional.of(dependencyByGroupIdAndArtifactId));
                return true;
            }
            return false;
        });
        return dependencyOptional.get();
    }

    /**
     * Traverses pom.xml files in the project hierarchy.
     *
     * @param pomContainingFolder
     *            the folder containing the pom.xml file
     * @param projectRootFolder
     *            the root directory of the project
     * @param function
     *            the function to apply to each POM file rewriter
     */
    private void walkThroughPomFiles(Path pomContainingFolder, Path projectRootFolder,
            Function<PomFileRewriter, Boolean> function) {
        Path pomFilePath = Path.of(pomContainingFolder.toString(), "pom.xml");
        // do not go up from project root folder
        if (pomContainingFolder.equals(projectRootFolder.getParent())) {
            return;
        }
        try {
            PomFileRewriter pomFileRewriter = new PomFileRewriter(pomFilePath.toFile());
            Boolean result = function.apply(pomFileRewriter);
            if (Boolean.TRUE.equals(result)) {
                return;
            }
            if (pomFileRewriter.hasParentPom()) {
                walkThroughPomFiles(Path.of(pomContainingFolder.getParent().toString()), projectRootFolder, function);
            }
        } catch (Exception e) {
            throw new CopilotException("Could not read pom.xml file", e);
        }
    }

    /**
     * Updates the Vaadin version property in the pom.xml file if it is present.
     *
     * @param propertyKey
     *            the key of the property to update
     * @param value
     *            the new version value to set
     * @param pomContainingFolder
     *            the directory containing the pom.xml file
     * @param projectRootFolder
     *            the root directory of the project
     * @return {@code true} if the update was successful, {@code false} otherwise
     */
    private boolean updateVaadinVersionPropertyInPomFileIfPresent(String propertyKey, String value,
            Path pomContainingFolder, Path projectRootFolder) {
        AtomicBoolean updateResultAtomicReference = new AtomicBoolean(false);
        walkThroughPomFiles(pomContainingFolder, projectRootFolder, model -> {
            String normalizedPropertyKey = propertyKey;
            if (propertyKey.startsWith("${") && propertyKey.endsWith("}")) {
                normalizedPropertyKey = propertyKey.substring(2, propertyKey.length() - 1);
            }
            PomFileRewriter.Property property = model.findPropertyByKey(normalizedPropertyKey);
            if (property != null) {
                property.fileRewriter().updateProperty(property, value);
                try {
                    property.fileRewriter().save();
                    updateResultAtomicReference.set(true);
                    return true;
                } catch (Exception e) {
                    throw new CopilotException("Could not update property", e);
                }
            }
            return false;
        });
        return updateResultAtomicReference.get();
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
