package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.vaadin.flow.component.UI;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.shared.util.SharedUtil;

import com.fasterxml.jackson.databind.JsonNode;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Util {

    private static final Pattern endsWithNumber = Pattern.compile("([0-9]+)$");

    private static Logger getLogger() {
        return LoggerFactory.getLogger(Util.class);
    }

    /**
     * Takes a filename and increases a trailing number on the name (excluding the
     * extension).
     *
     * <p>
     * For example, "file1.txt" would return "file2.txt", "file.txt" would return
     * "file1.txt" and "file123.txt" would return "file124.txt". If the filename
     * doesn't end with a number, a 1 is added at the end.
     *
     * @param filename
     *            the string to increase
     * @return the input string with the trailing number increased by one, or with a
     *         1 added at the end if it didn't end with a number
     */
    public static String increaseTrailingNumber(String filename) {
        String[] parts = filename.split("\\.", 2);
        String base = parts[0];
        String extension = parts[1];

        Matcher matcher = endsWithNumber.matcher(base);
        if (matcher.find()) {
            return base.substring(0, matcher.start()) + (Integer.parseInt(matcher.group()) + 1) + "." + extension;
        }
        return base + "1." + extension;
    }

    /**
     * Finds the first folder inside the given folder that contains more than one
     * sub-folder.
     *
     * <p>
     *
     * @param javaSourceFolder
     *            the root java source folder
     * @return the first folder that contains more than one sub-folder
     */
    public static File getSinglePackage(File javaSourceFolder) {
        File[] files = javaSourceFolder.listFiles();
        if (files == null || files.length == 0 || files.length > 1) {
            // Empty folder or folder containing multiple files
            return javaSourceFolder;
        }

        if (files[0].isDirectory()) {
            return getSinglePackage(files[0]);
        }
        return javaSourceFolder;
    }

    /**
     * Converts a string to title case.
     *
     * <p>
     * For example, "hello world" becomes "Hello World".
     *
     * @param input
     *            the input string
     * @return the input string converted to title case
     */
    public static String titleCase(String input) {
        return Arrays.stream(input.split(" ")).map(SharedUtil::capitalize).collect(Collectors.joining(" "));
    }

    /**
     * Replaces any folder with the given name in the path with a new folder name.
     *
     * @param path
     *            the path to modify
     * @param oldFolderName
     *            the name of the folder to replace
     * @param newFolderName
     *            the name of the folder to replace with
     * @param parentFolder
     *            the name of the folder containing the folder to replace
     * @return the modified path
     */
    public static Path replaceFolderInPath(Path path, String oldFolderName, String newFolderName, String parentFolder) {

        Path newPath = path.isAbsolute() ? Path.of("/") : Path.of("");

        boolean inParent = false;

        // Iterate over the path components
        for (Path part : path) {
            // If the current part matches the old folder name, replace it
            if (inParent && part.toString().equals(oldFolderName)) {
                newPath = newPath.resolve(newFolderName);
            } else {
                newPath = newPath.resolve(part);
            }
            inParent = parentFolder.equals(part.toString());
        }
        return newPath;
    }

    /**
     * Finds the common ancestor path for the given paths.
     * <p>
     * For instance if you pass in /foo/bar/baz and /foo/bar, this returns /foo/bar
     *
     * @param paths
     *            the paths to process
     * @return the common ancestor path
     */
    public static Optional<Path> findCommonAncestor(List<Path> paths) {
        if (paths == null || paths.isEmpty()) {
            return Optional.empty();
        }
        Path commonPath = paths.get(0);
        for (int i = 1; i < paths.size(); i++) {
            Path otherPath = paths.get(i);
            commonPath = Path.of(StringUtils.getCommonPrefix(commonPath.toString(), otherPath.toString()));
        }
        return Optional.of(commonPath);
    }

    /**
     * Finds a file related to the current view, to be able to determine which
     * folder / project module to create new files in.
     *
     * @param session
     *            the vaadin session to use
     * @param currentView
     *            JSON data for the current view, with either "viewFile" pointing to
     *            a Hilla view file or "uiId" refering to an open Flow UI
     * @return a file (Java or TSX) used for the current view
     */
    public static Optional<File> findCurrentViewFile(VaadinSession session, JsonNode currentView) {
        if (currentView == null) {
            return Optional.empty();
        }
        if (currentView.has("views") && !currentView.withArray("views").isEmpty()) {
            return Optional.of(new File(currentView.withArray("views").get(0).asText()));
        }
        if (currentView.has("uiId")) {
            int uiId = currentView.get("uiId").asInt();
            session.lock();
            try {
                UI ui = session.getUIById(uiId);
                return new ComponentSourceFinder(session)._getSourceLocation(ui.getCurrentView()).javaFile();
            } finally {
                session.unlock();
            }
        }
        return Optional.empty();
    }

    /**
     * Finds the module where the (first of) currently open view(s) is defined.
     *
     * @param session
     *            the vaadin session to use
     * @param currentView
     *            JSON data for the current view, with either "viewFile" pointing to
     *            a Hilla view file or "uiId" referring to an open Flow UI
     * @return the module where the current view is defined
     */
    public static Optional<JavaSourcePathDetector.ModuleInfo> findCurrentModule(VaadinSession session,
            JsonNode currentView) {
        return findCurrentViewFile(session, currentView).flatMap(ProjectFileManager.get()::findModule);
    }

    /**
     * Generates the getter method name for a given property.
     *
     * @param propertyName
     *            the name of the property
     * @param javaType
     *            the Java type of the property
     * @return the getter method name
     */
    public static String getGetterName(String propertyName, String javaType) {
        String prefix = javaType.equals(boolean.class.getName()) ? "is" : "get";
        return prefix + SharedUtil.capitalize(propertyName);
    }

    /**
     * Generates the setter method name for a given property.
     *
     * @param propertyName
     *            the name of the property
     * @return the setter method name
     */
    public static String getSetterName(String propertyName) {
        return "set" + SharedUtil.capitalize(propertyName);
    }

    /**
     * Converts a fully qualified class name to a filename.
     *
     * @param className
     *            the fully qualified class name
     * @return the filename corresponding to the class name
     */
    public static String getFilenameFromClassName(String className) {
        String packageName = getPackageName(className);
        String ret = "";
        String rest;
        if (!packageName.isEmpty()) {
            ret += packageName.replace(".", File.separator) + File.separator;
            rest = className.replace(packageName + ".", "");
        } else {
            rest = className;
        }
        if (rest.contains("$")) {
            rest = rest.substring(0, rest.indexOf("$"));
        }
        ret += rest + ".java";
        return ret;
    }

    public static String getClassNameFromFilename(Path filename, Path sourceFolder) {
        if (filename == null || filename.equals(sourceFolder)) {
            return "";
        }
        if (Files.isDirectory(filename)) {
            String directoryName = sourceFolder.relativize(filename).toString();
            return directoryName.replace(File.separator, ".");
        }
        String pkg = getClassNameFromFilename(filename.getParent(), sourceFolder);
        if (!pkg.isEmpty()) {
            pkg += ".";
        }
        return pkg + FilenameUtils.removeExtension(filename.getFileName().toString());
    }

    /**
     * Extracts the package name from a fully qualified class name.
     *
     * @param className
     *            the fully qualified class name
     * @return the package name
     */
    public static String getPackageName(String className) {
        int lastDot = className.lastIndexOf(".");
        if (lastDot == -1) {
            return "";
        }
        return className.substring(0, lastDot);
    }

    /**
     * Extracts the simple class name from a fully qualified class name.
     *
     * @param className
     *            the fully qualified class name
     * @return the simple class name
     */
    public static String getSimpleName(String className) {
        int lastDot = Math.max(className.lastIndexOf("."), className.lastIndexOf("$"));
        if (lastDot == -1) {
            return className;
        }
        return className.substring(lastDot + 1);
    }

    public static String findFeaturePackage(String viewPackage, String mainPackage) {
        if (viewPackage.startsWith(mainPackage + ".") && viewPackage.endsWith(".ui.view")) {
            // Feature package is the main package + the following package
            return mainPackage + "." + viewPackage.substring(mainPackage.length() + 1).split("\\.")[0];
        }
        return null;
    }

    public static List<String> getJavaPackages(Path sourceFolder) {
        return Util.listAllFolders(sourceFolder).stream()
                .map(folder -> Util.getClassNameFromFilename(folder, sourceFolder)).toList();
    }

    public static List<Path> listAllFolders(Path startPath) {
        List<Path> folderList = new ArrayList<>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(startPath)) {
            for (Path entry : stream) {
                if (Files.isDirectory(entry)) {
                    folderList.add(entry);
                    folderList.addAll(listAllFolders(entry));
                }
            }
        } catch (IOException e) {
            getLogger().warn("Unable to list {}", startPath, e);
        }
        return folderList;
    }

    private static String decidePackage(JavaSourcePathDetector.ModuleInfo moduleInfo, Path referenceFile,
            String mainPackage, String featurePackageName, String... alternativePackages) {
        Path javaSourcePath = moduleInfo.javaSourcePaths().get(0);
        String referenceClass = Util.getClassNameFromFilename(referenceFile, javaSourcePath);
        if (!referenceClass.isEmpty()) {
            String referencePackage = Util.getPackageName(referenceClass);

            // When using "package by feature" we want to find the feature package and start
            // from there
            String featurePackage = Util.findFeaturePackage(referencePackage, mainPackage);
            if (featurePackage != null) {
                return featurePackage + "." + featurePackageName;
            }
        }

        List<String> packages = Util.getJavaPackages(javaSourcePath);
        Stream<String> candidates = Stream.empty();
        for (String alternativePackage : alternativePackages) {
            candidates = Stream.concat(candidates, findPackages(packages, alternativePackage));
        }
        return candidates.findFirst().orElseGet(() -> mainPackage + "." + featurePackageName);
    }

    public static String decideEntityPackage(JavaSourcePathDetector.ModuleInfo moduleInfo, Path referenceFile,
            String mainPackage) {
        return decidePackage(moduleInfo, referenceFile, mainPackage, "domain", "entity", "data", "domain");
    }

    public static String decideRepositoryPackage(JavaSourcePathDetector.ModuleInfo moduleInfo, Path referenceFile,
            String mainPackage) {
        return decidePackage(moduleInfo, referenceFile, mainPackage, "domain", "repository", "repositories", "data",
                "domain");
    }

    public static String decideServicePackage(JavaSourcePathDetector.ModuleInfo moduleInfo, Path referenceFile,
            String mainPackage) {
        return decidePackage(moduleInfo, referenceFile, mainPackage, "service", "services", "service");
    }

    private static Stream<String> findPackages(List<String> javaPackages, String endsWith) {
        return javaPackages.stream().filter(javaPackage -> javaPackage.endsWith("." + endsWith));
    }

    /**
     * Gets the parameter types of a method.
     * 
     * @param method
     *            the method to get parameter types for
     * @return a list of parameter types
     */
    public static List<JavaReflectionUtil.ParameterTypeInfo> getParameterTypes(Method method) {
        Parameter[] parameters = method.getParameters();
        return Arrays.stream(parameters).map(parameter -> new JavaReflectionUtil.ParameterTypeInfo(parameter.getName(),
                new JavaReflectionUtil.TypeInfo(parameter.getType().getName(), List.of()))).toList();
    }

    /**
     * Returns the name of the file, relative to the given folder.
     *
     * @param projectFile
     *            the file
     * @param folder
     *            the folder to make relative to
     * @return the relative name of the file
     */
    public static String getRelativeName(File projectFile, File folder) {
        return folder.toPath().relativize(projectFile.toPath()).toString();
    }

    /**
     * Checks if a file is inside a container folder.
     *
     * @param toCheck
     *            the file to check
     * @param container
     *            the container folder
     * @return true if the file is inside the container folder
     */
    public static boolean isFileInside(File toCheck, File container) {
        if (!container.exists()) {
            return false;
        }
        if (toCheck.exists()) {
            Path pathToCheck = toCheck.toPath();
            try {
                return pathToCheck.toRealPath().startsWith(container.toPath().toRealPath().toString());
            } catch (IOException e) {
                getLogger().error("Unable to check if " + toCheck + " is inside " + container, e);
                return false;
            }
        } else {
            // New file
            return isFileInside(toCheck.getParentFile(), container);
        }
    }

    /**
     * Inserts the given string on the given line numbers in the given text
     */
    public static String insertLines(String text, List<Integer> lineNumbers, String toInsert) {
        if (text == null) {
            return null;
        }
        if (text.isEmpty() || lineNumbers == null || lineNumbers.isEmpty() || toInsert == null) {
            return text;
        }

        String[] lines = text.split("\\R", -1);
        StringBuilder result = new StringBuilder();

        for (int i = 0; i < lines.length; i++) {
            // Line numbers in the list are 1-based, but array indices are 0-based
            if (lineNumbers.contains(i + 1)) {
                // Insert the comment before the line
                result.append(toInsert).append("\n");
            }
            result.append(lines[i]);

            // Add newline if not the last line
            if (i < lines.length - 1) {
                result.append("\n");
            }
        }

        return result.toString();
    }

    public static String truncate(String str, int i) {
        return str.substring(0, Math.max(0, i - str.length()));
    }

    public static String escapeSingleQuote(String value) {
        return value.replace("'", "\\'");
    }
}
