package com.vaadin.copilot;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;

import com.vaadin.flow.server.frontend.ProxyFactory;
import com.vaadin.flow.server.frontend.installer.DefaultFileDownloader;
import com.vaadin.flow.server.frontend.installer.DownloadException;
import com.vaadin.flow.server.frontend.installer.FileDownloader;
import com.vaadin.flow.server.frontend.installer.ProxyConfig;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Downloader {

    private Downloader() {
        // Util class
    }

    /**
     * Downloads a file from the given URL to the target file, taking proxies into
     * account.
     *
     * @param url
     *            the URL to download from
     * @param target
     *            the target file to download to
     * @param progressListener
     *            the progress listener to use for tracking the download progress
     */
    public static void downloadFile(URL url, File target, FileDownloader.ProgressListener progressListener)
            throws URISyntaxException, DownloadException {
        ProxyConfig proxyConfig = new ProxyConfig(ProxyFactory.getProxies(ProjectFileManager.get().getProjectRoot()));
        DefaultFileDownloader downloader = new DefaultFileDownloader(proxyConfig);
        downloader.download(url.toURI(), target, null, null, progressListener);
    }

    /**
     * Downloads the given URL and provide the data as a stream.
     *
     * @param url
     *            the URL to download from
     * @param progressListener
     *            the progress listener to use for tracking the download progress
     */
    public static InputStream download(URL url, FileDownloader.ProgressListener progressListener)
            throws URISyntaxException, DownloadException, IOException {
        File tempFile = Files.createTempFile("copilot-temp", ".tmp").toFile();
        downloadFile(url, tempFile, progressListener);
        return new FileInputStream(tempFile) {
            @Override
            public void close() throws IOException {
                super.close();
                if (!tempFile.delete()) {
                    getLogger().warn("Unable to delete temporary file {}", tempFile.getAbsolutePath());
                }
            }
        };
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(Downloader.class);
    }
}
