package com.vaadin.copilot.javarewriter.custom;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.FlowComponentQuirks;
import com.vaadin.copilot.javarewriter.InsertionPoint;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.copilot.javarewriter.JavaSource;

import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.ClassExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithArguments;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.type.ClassOrInterfaceType;

/**
 * Crud component handler.
 */
public class CrudComponentHandle extends CustomComponentHandle {

    @Override
    public List<VariableDeclarator> createComponentStatements(JavaRewriter javaRewriter, JavaComponent component,
            InsertionPoint insertionPoint, JavaComponent parent, String layoutVariableName,
            ComponentInfo referenceComponent, JavaRewriter.AddTemplateOptions options, JavaSource javaSource) {

        String componentClassName = FlowComponentQuirks.getClassForComponent(component);
        ClassOrInterfaceType fullType = StaticJavaParser.parseClassOrInterfaceType(componentClassName); // Import

        CompilationUnit compilationUnit = insertionPoint.getCompilationUnit();
        JavaRewriterUtil.addImport(compilationUnit, fullType.getNameWithScope());
        JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.component.crud.BinderCrudEditor", false, false);
        JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.data.binder.Binder", false, false);

        JavaComponent grid = component.withTag("Grid");
        ComponentInfo gridComponent = javaRewriter.createComponentStatements(insertionPoint, parent, grid, false,
                layoutVariableName, referenceComponent, options, null, javaSource);
        if (gridComponent == null) {
            throw new IllegalArgumentException("Could not generate Grid for the Crud component");
        }
        VariableDeclarator gridVariableDeclarator = gridComponent.getVariableDeclarator();
        String gridVariableName = gridVariableDeclarator.getNameAsString();
        String crudVariableName = JavaRewriterUtil.generateVariableName(component, fullType, insertionPoint);

        ClassExpr beanClassExpression = getBeanClassExpressionFromGrid(gridVariableDeclarator);

        fullType.setTypeArguments(beanClassExpression.getType());
        VariableDeclarator decl = new VariableDeclarator(fullType.removeScope(), crudVariableName);

        ObjectCreationExpr expr = new ObjectCreationExpr();
        expr.setType("Crud");

        ClassOrInterfaceType toDeleteType = new ClassOrInterfaceType();
        toDeleteType.setName("DELETE_THIS");
        expr.getType().setTypeArguments(toDeleteType);
        decl.setInitializer(expr);

        expr.getArguments().add(beanClassExpression);
        expr.getArguments().add(new NameExpr(gridVariableName));
        expr.getArguments().add(new NameExpr("new BinderCrudEditor<>(new Binder<>())"));

        VariableDeclarationExpr declarationExpr = new VariableDeclarationExpr(decl);

        insertionPoint.add(new ExpressionStmt(declarationExpr));
        javaRewriter.attachComponent(insertionPoint, component, parent, layoutVariableName, referenceComponent,
                new NameExpr(crudVariableName), crudVariableName, options, null);

        return new ArrayList<>();
    }

    private ClassExpr getBeanClassExpressionFromGrid(VariableDeclarator declarator) {
        Optional<Expression> initializer = declarator.getInitializer();
        if (initializer.isPresent()) {
            Expression expression = initializer.get();
            if (expression instanceof NodeWithArguments<?> nodeWithArguments) {
                for (Expression argument : nodeWithArguments.getArguments()) {
                    if (argument instanceof ClassExpr classExpr) {
                        return classExpr;
                    }
                }
            }
        }
        throw new IllegalArgumentException("Unable to locate bean type of the grid");
    }
}
