/**
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.flow.component;

import java.util.stream.Stream;

/**
 * Possible keyboard key locations.
 *
 * The location attribute can be used to disambiguate between key values that
 * can be generated by different physical keys on the keyboard, for example, the
 * left and right Shift key.
 *
 * @since 1.0
 */
public enum KeyLocation {

    /**
     * Standard key location.
     */
    STANDARD(0),

    /**
     * Left key location.
     */
    LEFT(1),

    /**
     * Right key location.
     */
    RIGHT(2),

    /**
     * Numeric pad key location.
     */
    NUMPAD(3);

    private final int location;

    KeyLocation(int location) {
        this.location = location;
    }

    /**
     * Gets the key location integer value.
     *
     * @return the key location integer value
     */
    public int getLocation() {
        return location;
    }

    /**
     * Returns the {@code KeyLocation} for {@code location}.
     *
     * @param location
     *            the location integer value
     * @return the {@code KeyLocation}
     */
    public static KeyLocation of(int location) {
        return Stream.of(values()).filter(l -> l.location == location)
                .findFirst().orElseThrow(IllegalArgumentException::new);
    }

}
