/**
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.flow.theme;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Defines the theme to use for the application. Should be present only once for
 * an app, in the `AppShellConfiguration` class.
 * <p>
 * The {@code value} property defines the name of the application theme. When
 * the theme is present inside the project, it maps to the {@code
 * frontend/theme/<application-theme-name>}/ folder or alternatively to a folder
 * inside the static resources of a jar file, like {@code
 * src/main/resources/META-INF/resources/theme/<application-theme-name>/}. The
 * application theme is always based on Lumo theme and this is the recommended
 * way to theme applications starting from Flow 6.0 and Vaadin 19.
 * <p>
 * Alternatively , if instead of Lumo theme the Material theme or another "old
 * style custom theme" is to be used, that can be specified with the {@code
 * themeClass} property. This allows defining a way to translate base component
 * url to the themed component url (@see {@link AbstractTheme}), which specifies
 * components styles.
 * <p>
 * By default {@code com.vaadin.flow.theme.lumo.Lumo} theme is used if it's in
 * the classpath. You may disable theming with {@link NoTheme} annotation.
 * <p>
 * {@link Theme} annotation should be added to the AppShellConfigurator
 * implementation.
 *
 * <p>
 * Only a single theme can be defined and having multiple instances will throw
 * an exception.
 *
 * <p>
 * Here is some examples:
 *
 * <pre>
 * <code>
 *
 * &#64;Theme("my-theme")
 * public class MyAppShell implements AppShellConfigurator {
 * }
 * </code>
 * </pre>
 *
 * <pre>
 * <code>
 *
 * &#64;Theme(themeClass = Lumo.class)
 * public class MyAppShell implements AppShellConfigurator {
 * }
 * </code>
 * </pre>
 *
 * @author Vaadin Ltd
 * @see AbstractTheme
 * @see NoTheme
 * @since 1.0
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.RUNTIME)
@Inherited
@Documented
public @interface Theme {

    /**
     * The theme translation handler.
     *
     * Defaults to Lumo, If not specified.
     *
     * @return theme handler
     */
    Class<? extends AbstractTheme> themeClass() default AbstractTheme.class;

    /**
     * The theme variant, if any.
     *
     * @return the theme variant
     */
    String variant() default "";

    /**
     * The name of the theme to use.
     *
     * If this is not specified will default to Lumo.
     */
    String value() default "";
}
