/**
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.flow.data.provider;

import java.util.Comparator;
import java.util.Locale;
import java.util.Objects;

import com.vaadin.flow.component.UI;
import com.vaadin.flow.function.SerializableBiPredicate;
import com.vaadin.flow.function.SerializableComparator;
import com.vaadin.flow.function.SerializablePredicate;
import com.vaadin.flow.function.SerializableSupplier;
import com.vaadin.flow.function.ValueProvider;

/**
 * Helper methods for implementing {@link InMemoryDataProvider}s.
 * <p>
 * This class is intended primarily for internal use.
 *
 * @author Vaadin Ltd
 * @since 1.0
 */
public class InMemoryDataProviderHelpers {

    /**
     * Supplier that attempts to resolve a locale from the current UI. Returns
     * the system's default locale as a fallback.
     */
    public static final SerializableSupplier<Locale> CURRENT_LOCALE_SUPPLIER = () -> {
        UI currentUi = UI.getCurrent();
        if (currentUi != null) {
            return currentUi.getLocale();
        } else {
            return Locale.getDefault();
        }
    };

    private InMemoryDataProviderHelpers() {
    }

    /**
     * Wraps a given data provider so that its filter ignores null items
     * returned by the given value provider.
     *
     * @param dataProvider
     *            the data provider to wrap
     * @param valueProvider
     *            the value provider for providing values to filter
     * @param predicate
     *            the predicate to combine null filtering with
     * @param <T>
     *            the provided data type
     * @param <Q>
     *            the filter type
     * @param <V>
     *            the data provider object type
     * @return the wrapped data provider
     */
    public static <T, V, Q> DataProvider<T, Q> filteringByIgnoreNull(
            InMemoryDataProvider<T> dataProvider,
            ValueProvider<T, V> valueProvider,
            SerializableBiPredicate<V, Q> predicate) {
        Objects.requireNonNull(predicate, "Predicate cannot be null");

        return dataProvider.filteringBy(valueProvider,
                (itemValue, queryFilter) -> itemValue != null
                        && predicate.test(itemValue, queryFilter));
    }

    /**
     * Wraps a given data provider so that its filter tests the given predicate
     * with the lower case string provided by the given value provider.
     *
     * @param dataProvider
     *            the data provider to wrap
     * @param valueProvider
     *            the value provider for providing string values to filter
     * @param predicate
     *            the predicate to use for comparing the resulting lower case
     *            strings
     * @param localeSupplier
     *            the locale to use when converting strings to lower case
     * @param <T>
     *            the data provider object type
     * @return the wrapped data provider
     */
    public static <T> DataProvider<T, String> filteringByCaseInsensitiveString(
            InMemoryDataProvider<T> dataProvider,
            ValueProvider<T, String> valueProvider,
            SerializableBiPredicate<String, String> predicate,
            SerializableSupplier<Locale> localeSupplier) {
        // Only assert since these are only passed from our own code
        assert predicate != null;
        assert localeSupplier != null;

        return filteringByIgnoreNull(dataProvider, valueProvider,
                (itemString, filterString) -> {
                    Locale locale = localeSupplier.get();
                    assert locale != null;

                    return predicate.test(itemString.toLowerCase(locale),
                            filterString.toLowerCase(locale));
                });
    }

    /**
     * Creates a comparator for the return type of the given
     * {@link ValueProvider}, sorted in the direction specified by the given
     * {@link SortDirection}.
     *
     * @param valueProvider
     *            the value provider to use
     * @param sortDirection
     *            the sort direction to use
     * @param <T>
     *            the data provider object type
     * @param <V>
     *            the provided value type
     *
     * @return the created comparator
     */
    public static <V extends Comparable<? super V>, T> SerializableComparator<T> propertyComparator(
            ValueProvider<T, V> valueProvider, SortDirection sortDirection) {
        Objects.requireNonNull(valueProvider, "Value provider cannot be null");
        Objects.requireNonNull(sortDirection, "Sort direction cannot be null");

        Comparator<V> comparator = getNaturalSortComparator(sortDirection);

        return (a, b) -> comparator.compare(valueProvider.apply(a),
                valueProvider.apply(b));
    }

    /**
     * Gets the natural order comparator for the type argument, or the natural
     * order comparator reversed if the given sorting direction is
     * {@link SortDirection#DESCENDING}.
     *
     * @param sortDirection
     *            the sort direction to use
     * @param <V>
     *            the objects to compare
     * @return the natural comparator, with ordering defined by the given sort
     *         direction
     */
    public static <V extends Comparable<? super V>> Comparator<V> getNaturalSortComparator(
            SortDirection sortDirection) {
        Comparator<V> comparator = Comparator.naturalOrder();
        if (sortDirection == SortDirection.DESCENDING) {
            comparator = comparator.reversed();
        }
        return comparator;
    }

    /**
     * Creates a new predicate from the given predicate and value provider. This
     * allows using a predicate of the value providers return type with objects
     * of the value providers type.
     *
     * @param valueProvider
     *            the value provider to use
     * @param valueFilter
     *            the original predicate
     * @param <T>
     *            the data provider object type
     * @param <V>
     *            the provided value type
     * 
     * @return the created predicate
     */
    public static <T, V> SerializablePredicate<T> createValueProviderFilter(
            ValueProvider<T, V> valueProvider,
            SerializablePredicate<V> valueFilter) {
        return item -> valueFilter.test(valueProvider.apply(item));
    }

    /**
     * Creates a predicate that compares equality of the given required value to
     * the value the given value provider obtains.
     *
     * @param valueProvider
     *            the value provider to use
     * @param requiredValue
     *            the required value
     * @param <T>
     *            the data provider object type
     * @param <V>
     *            the provided value type
     * @return the created predicate
     */
    public static <T, V> SerializablePredicate<T> createEqualsFilter(
            ValueProvider<T, V> valueProvider, V requiredValue) {
        Objects.requireNonNull(valueProvider, "Value provider cannot be null");

        return item -> Objects.equals(valueProvider.apply(item), requiredValue);
    }
}
