/**
 * Copyright (C) 2000-2023 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.fusion.auth;

import javax.servlet.http.Cookie;
import java.util.Arrays;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Stream;

import com.vaadin.flow.internal.springcsrf.SpringCsrfTokenUtil;
import com.vaadin.flow.server.VaadinRequest;
import com.vaadin.flow.server.VaadinResponse;
import com.vaadin.flow.server.communication.IndexHtmlRequestListener;
import com.vaadin.flow.server.communication.IndexHtmlResponse;
import com.vaadin.flow.shared.ApplicationConstants;

/**
 * An index HTML request listener that generates and sends a token for
 * Cross-Site Request Forgery protection (Double Submit Cookie pattern) of
 * Fusion endpoints.
 *
 * Sets a JS readable cookie in the response with the CSRF token, if such a
 * cookie was not present in request.
 *
 * @see CsrfChecker
 */
public class CsrfIndexHtmlRequestListener implements IndexHtmlRequestListener {
    @Override
    public void modifyIndexHtmlResponse(IndexHtmlResponse indexHtmlResponse) {
        ensureCsrfTokenCookieIsSet(indexHtmlResponse.getVaadinRequest(),
                indexHtmlResponse.getVaadinResponse());
    }

    private void ensureCsrfTokenCookieIsSet(VaadinRequest request,
            VaadinResponse response) {
        if (isSpringCsrfTokenPresent(request)) {
            return;
        }
        final String csrfCookieValue = Optional.ofNullable(request.getCookies())
                .map(Arrays::stream).orElse(Stream.empty())
                .filter(cookie -> cookie.getName()
                        .equals(ApplicationConstants.CSRF_TOKEN))
                .findFirst().map(Cookie::getValue).orElse(null);
        if (csrfCookieValue != null && !csrfCookieValue.isEmpty()) {
            return;
        }

        /*
         * Despite section 6 of RFC 4122, this particular use of UUID *is*
         * adequate for security capabilities. Type 4 UUIDs contain 122 bits of
         * random data, and UUID.randomUUID() is defined to use a
         * cryptographically secure random generator.
         */
        final String csrfToken = UUID.randomUUID().toString();
        Cookie csrfCookie = new Cookie(ApplicationConstants.CSRF_TOKEN,
                csrfToken);
        csrfCookie.setSecure(request.isSecure());
        String path = request.getContextPath();
        if (path == null || path.isEmpty()) {
            path = "/";
        }
        csrfCookie.setPath(path);
        csrfCookie.setHttpOnly(false);
        response.addCookie(csrfCookie);
    }

    boolean isSpringCsrfTokenPresent(VaadinRequest request) {
        return SpringCsrfTokenUtil.getSpringCsrfToken(request).isPresent();
    }
}
