/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.hilla.route;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.AntPathMatcher;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;

import com.vaadin.flow.function.DeploymentConfiguration;
import com.vaadin.flow.router.internal.ClientRoutesProvider;
import com.vaadin.flow.server.frontend.FrontendUtils;
import com.vaadin.hilla.route.records.ClientViewConfig;

/**
 * Keeps track of registered client side routes.
 */
@Component
public class ClientRouteRegistry implements ClientRoutesProvider {

    public static final String FILE_ROUTES_JSON_NAME = "file-routes.json";
    public static final String FILE_ROUTES_JSON_PROD_PATH = "/META-INF/VAADIN/"
            + FILE_ROUTES_JSON_NAME;

    /**
     * A map of registered routes and their corresponding client view
     * configurations with ordered insertion.
     */
    private final Map<String, ClientViewConfig> registeredRoutes = new LinkedHashMap<>();

    private final ObjectMapper mapper = new ObjectMapper();

    private static final Logger LOGGER = LoggerFactory
            .getLogger(ClientRouteRegistry.class);

    private volatile LocalDateTime lastUpdated;

    private volatile boolean hasMainLayout;

    /**
     * Returns all registered routes.
     *
     * @return a map of all registered routes
     */
    public synchronized Map<String, ClientViewConfig> getAllRoutes() {
        return Map.copyOf(registeredRoutes);
    }

    /**
     * Clears all registered routes. For internal use only.
     */
    synchronized void clearRoutes() {
        hasMainLayout = false;
        registeredRoutes.clear();
    }

    /**
     * Adds a new route to the registry. For internal use only.
     *
     * @param route
     *            the route to add
     * @param clientView
     *            the client view to add
     * @throws IllegalStateException
     *             if the route has already been registered
     *
     */
    synchronized void addRoute(String route, ClientViewConfig clientView) {
        if (registeredRoutes.containsKey(route)) {
            throw new IllegalStateException(
                    "An attempt to register a route that is already registered. Route: "
                            + route + " View: " + clientView.getRoute());
        }
        registeredRoutes.put(route, clientView);
    }

    /**
     * Gets the client view configuration for the given route.
     *
     * @param path
     *            the URL path to get the client view configuration for
     * @return - the client view configuration for the given route
     */
    public synchronized ClientViewConfig getRouteByPath(String path) {
        final Set<String> routes = registeredRoutes.keySet();
        final AntPathMatcher pathMatcher = new AntPathMatcher();
        return Stream.of(addTrailingSlash(path), removeTrailingSlash(path))
                .map(p -> {
                    for (String route : routes) {
                        if (pathMatcher.match(route, p)) {
                            return registeredRoutes.get(route);
                        }
                    }
                    return null;
                }).filter(Objects::nonNull).findFirst().orElse(null);
    }

    private String addTrailingSlash(String path) {
        return path.endsWith("/") ? path : path + '/';
    }

    private String removeTrailingSlash(String path) {
        return path.endsWith("/") ? path.substring(0, path.length() - 1) : path;
    }

    /**
     * Registers client routes from file-routes.json file generated by the
     * file-router's Vite plugin. The file-routes.json file is expected to be in
     * the frontend/generated folder in dev mode and in the META-INF/VAADIN
     * folder in production mode.
     *
     * @param deploymentConfiguration
     *            the deployment configuration
     * @param lastUpdated
     *            the time of latest loading of the file-routes.json
     */
    public synchronized void registerClientRoutes(
            DeploymentConfiguration deploymentConfiguration,
            LocalDateTime lastUpdated) {
        var fileRoutesJsonAsResource = getFileRoutesJsonAsResource(
                deploymentConfiguration);
        if (fileRoutesJsonAsResource == null) {
            LOGGER.debug(
                    "No {} found under {} directory. Skipping client route registration.",
                    FILE_ROUTES_JSON_NAME,
                    deploymentConfiguration.isProductionMode()
                            ? "'META-INF/VAADIN'"
                            : "'frontend/generated'");
            return;
        }
        try (var source = fileRoutesJsonAsResource.openStream()) {
            if (source != null) {
                clearRoutes();
                mapper.readValue(source,
                        new TypeReference<List<ClientViewConfig>>() {
                        }).forEach(
                                route -> registerAndRecurseChildren("", route));
                this.lastUpdated = lastUpdated;
            }
        } catch (IOException e) {
            if (deploymentConfiguration.isProductionMode()) {
                // The file should be available in production mode:
                LOGGER.error("Failed to load {} from {}", FILE_ROUTES_JSON_NAME,
                        fileRoutesJsonAsResource.getPath(), e);
            } else {
                LOGGER.debug(
                        "Failed to load {} from {}. Skipping client route registration. "
                                + "There might be a problem with the contents of the file-routes.json file.",
                        FILE_ROUTES_JSON_NAME, "'frontend/generated'");
            }
        }
    }

    public synchronized void loadLatestDevModeFileRoutesJsonIfNeeded(
            DeploymentConfiguration deploymentConfiguration) {
        var devModeFileRoutesJsonFile = deploymentConfiguration
                .getFrontendFolder().toPath().resolve("generated")
                .resolve("file-routes.json").toFile();
        if (!devModeFileRoutesJsonFile.exists()) {
            LOGGER.debug("No file-routes.json found under {}",
                    deploymentConfiguration.getFrontendFolder().toPath()
                            .resolve("generated"));
            return;
        }
        var lastModified = devModeFileRoutesJsonFile.lastModified();
        var lastModifiedTime = Instant.ofEpochMilli(lastModified)
                .atZone(ZoneId.systemDefault()).toLocalDateTime();
        if (lastUpdated == null || lastModifiedTime.isAfter(lastUpdated)) {
            LOGGER.debug("Loading latest file-routes.json from dev mode");
            registerClientRoutes(deploymentConfiguration, lastModifiedTime);
        }
    }

    private URL getFileRoutesJsonAsResource(
            DeploymentConfiguration deploymentConfiguration) {
        var isProductionMode = deploymentConfiguration.isProductionMode();
        if (isProductionMode) {
            return getClass().getResource(FILE_ROUTES_JSON_PROD_PATH);
        }
        try {
            var fileRoutesJson = FrontendUtils
                    .getFrontendGeneratedFolder(
                            deploymentConfiguration.getFrontendFolder())
                    .toPath().resolve(FILE_ROUTES_JSON_NAME).toFile();
            if (!fileRoutesJson.exists()) {
                return null;
            } else {
                return fileRoutesJson.toURI().toURL();
            }
        } catch (MalformedURLException e) {
            LOGGER.error(
                    "Unexpected error while getting {} as resource from 'frontend/generated'.",
                    FILE_ROUTES_JSON_NAME, e);
            throw new RuntimeException(e);
        }
    }

    private void registerAndRecurseChildren(String basePath,
            ClientViewConfig view) {
        var path = view.getRoute() == null || view.getRoute().isEmpty()
                ? basePath
                : basePath + '/' + view.getRoute();

        if (!hasMainLayout && isMainLayout(view)) {
            hasMainLayout = true;
        }

        // Skip layout views without children.
        // https://github.com/vaadin/hilla/issues/2379
        if (view.getChildren() == null) {
            addRoute(path, view);
        } else {
            view.getChildren().forEach(child -> {
                child.setParent(view);
                registerAndRecurseChildren(path, child);
            });
        }
    }

    private boolean isMainLayout(ClientViewConfig view) {
        return (view.getRoute() == null || view.getRoute().isBlank())
                && view.getChildren() != null && view.getParent() == null;
    }

    /**
     * Gets whether the registry has a main layout.
     *
     * @return {@code true} if the registry has a main layout, {@code false}
     *         otherwise
     */
    public synchronized boolean hasMainLayout() {
        return hasMainLayout;
    }

    @Override
    public synchronized List<String> getClientRoutes() {
        return getAllRoutes().keySet().stream().toList();
    }
}
