package com.vaadin.pro.licensechecker;

import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.URL;
import java.net.URLConnection;
import java.net.UnknownHostException;
import java.util.logging.Level;
import java.util.logging.Logger;

import elemental.json.Json;
import elemental.json.JsonObject;

public class ProKeyValidator {

    private static final String UNABLE_TO_VALIDATE_SUBSCRIPTION = "Unable to validate subscription."
            + " Please go to https://vaadin.com/pro/validate-license to check that your subscription is active."
            + " You can also find instructions for installing the license on a build server at https://vaadin.com/pro/validate-license";
    private static final String LICENSE_VALIDATION_URL = "https://tools.vaadin.com/vaadin-license-server/licenses/pro";
    private static final String[] PROPERTIES = new String[] { "java.vendor",
            "java.version", "os.arch", "os.name", "os.version" };

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    public static void validate(Product product, ProKey proKey) {
        if (proKey == null) {
            getLogger().fine("No pro key found to validate");
            throw new RuntimeException(UNABLE_TO_VALIDATE_SUBSCRIPTION);
        }
        getLogger().fine("Validating pro key for " + product);
        if (History.isRecentlyValidated(product)) {
            // check only every 24h
            getLogger().fine(
                    "Skipping check as product license was recently validated.");
            return;
        }
        try {
            if (validateWithServer(product, proKey)) {
                History.setLastCheckTimeNow(product);
                return;
            } else {
                // Key does not have access to the given product
                // Subscription might have ended or some other problem
            }
        } catch (UnknownHostException | ConnectException e) {
            // Allow usage offline
            getLogger().log(Level.INFO,
                    "Unable to validate the license, please check your internet connection");
            return;
        } catch (Exception e) {
            throw new RuntimeException(UNABLE_TO_VALIDATE_SUBSCRIPTION, e);
        }

        throw new RuntimeException(UNABLE_TO_VALIDATE_SUBSCRIPTION);

    }

    private static boolean validateWithServer(Product product, ProKey proKey)
            throws IOException {
        getLogger().fine("Validating license for " + product + " with server");
        JsonObject json = queryServer(product, proKey);
        String result = json.getString("result");
        String message = json.getString("message");
        if (message != null && !message.isEmpty()) {
            getLogger().info(message);
        }
        return "ok".equals(result);
    }

    private static JsonObject queryServer(Product product, ProKey proKey)
            throws IOException {
        URL url = new URL(LICENSE_VALIDATION_URL);
        URLConnection http = url.openConnection();
        http.setRequestProperty("check-source", "java");
        http.setRequestProperty("product-name", product.getName());
        http.setRequestProperty("product-version", product.getVersion());
        for (String property : PROPERTIES) {
            String value = System.getProperty(property);
            if (value != null) {
                http.setRequestProperty("prop-" + property.replace(".", "-"),
                        value);
            }
        }

        http.setRequestProperty("product-version", product.getVersion());
        http.setRequestProperty("Cookie", "proKey=" + proKey.getProKey());
        http.connect();

        try (InputStream in = http.getInputStream()) {
            String response = Util.toString(in);
            return Json.parse(response);
        }
    }

}
