/**
 * Copyright (C) 2012 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import java.io.IOException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.ZoneOffset;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.pro.licensechecker.OnlineKeyValidator.Result;

public class LicenseChecker {

    private static boolean strictOffline = false;

    static String loggedLicenseOwner = null;

    public interface Callback {

        public void ok();

        public void failed(Exception e);
    }

    static Consumer<String> systemBrowserUrlHandler = url -> {
        try {
            getLogger().info("Opening system browser to validate license. If the browser is not opened, please open "
                    + url + " manually");
            SystemBrowser.open(url);
            getLogger().info("If you are working offline, please visit "
                    + OfflineKeyValidator.getOfflineUrl(MachineId.get()) + " for an offline license");
        } catch (IOException e) {
            getLogger().error(
                    "Error opening system browser to validate license. Please open " + url + " manually", e);
        }

    };

    /**
     * @deprecated use
     *             {@link #checkLicenseFromStaticBlock(String, String, BuildType)}
     */
    @Deprecated
    public static void checkLicenseFromStaticBlock(String productName,
            String productVersion) {
        checkLicenseFromStaticBlock(productName, productVersion, BuildType.DEVELOPMENT);
    }

    /**
     * Checks the license for the given product version from a {@code static}
     * block.
     * 
     * @param productName    the name of the product to check
     * @param productVersion the version of the product to check
     * @param buildType      the type of build: production or development
     * 
     * @throws ExceptionInInitializerError if the license check fails
     */
    public static void checkLicenseFromStaticBlock(String productName,
            String productVersion, BuildType buildType) {
        try {
            checkLicense(productName, productVersion, buildType);
        } catch (Exception e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    /**
     * @deprecated use {@link #checkLicense(String, String, BuildType)}
     */
    @Deprecated
    public static void checkLicense(String productName, String productVersion) {
        checkLicense(productName, productVersion, BuildType.DEVELOPMENT, systemBrowserUrlHandler);
    }

    /**
     * Checks the license for the given product version.
     * 
     * @param productName    the name of the product to check
     * @param productVersion the version of the product to check
     * @param buildType      the type of build: development or production
     * 
     * @throws LicenseException if the license check fails
     */
    public static void checkLicense(String productName, String productVersion, BuildType buildType) {
        checkLicense(productName, productVersion, buildType, systemBrowserUrlHandler);
    }

    /**
     * @deprecated use {@link #checkLicense(String, String, BuildType, Consumer)}
     */
    public static void checkLicense(String productName, String productVersion,
            Consumer<String> noKeyUrlHandler) {
        checkLicense(productName, productVersion, BuildType.DEVELOPMENT, noKeyUrlHandler);
    }

    /**
     * Checks the license for the given product version.
     * 
     * @param productName     the name of the product to check
     * @param productVersion  the version of the product to check
     * @param buildType       the type of build: development or production
     * @param noKeyUrlHandler a handler that is invoked to open the vaadin.com URL
     *                        to download the key file. Used when no key file is
     *                        avialable.
     * 
     * @throws LicenseException if the license check fails
     */
    public static void checkLicense(String productName, String productVersion, BuildType buildType,
            Consumer<String> noKeyUrlHandler) {
        checkLicense(new Product(productName, productVersion), buildType, noKeyUrlHandler);
    }

    /**
     * @deprecated use
     *             {@link #checkLicenseAsync(String, String, BuildType, Callback)}
     */
    @Deprecated
    public static void checkLicenseAsync(String productName,
            String productVersion, Callback callback) {
        checkLicenseAsync(productName, productVersion, BuildType.DEVELOPMENT, callback);
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done.
     * 
     * @param productName    the name of the product to check
     * @param productVersion the version of the product to check
     * @param buildType      the type of build: development or production
     * @param callback       the callback to invoke with the result
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback) {
        checkLicenseAsync(productName, productVersion, buildType, callback, systemBrowserUrlHandler);
    }

    /**
     * @deprecated use
     *             {@link #checkLicenseAsync(String, String, BuildType, Callback, Consumer)}
     */
    @Deprecated
    public static void checkLicenseAsync(String productName,
            String productVersion, Callback callback, Consumer<String> noKeyUrlHandler) {
    }

    /**
     * Checks the license for the given product version in the background and
     * invokes the callback when done.
     * 
     * @param productName     the name of the product to check
     * @param productVersion  the version of the product to check
     * @param buildType       the type of build: development or production
     * @param callback        the callback to invoke with the result
     * @param noKeyUrlHandler a handler that is invoked to open the vaadin.com URL
     *                        to download the key file. Used when no key file is
     *                        avialable.
     */
    public static void checkLicenseAsync(String productName,
            String productVersion, BuildType buildType, Callback callback, Consumer<String> noKeyUrlHandler) {
        checkLicenseAsync(new Product(productName, productVersion), buildType, callback, noKeyUrlHandler,
                MachineId.get(),
                LocalProKey.get(), LocalOfflineKey.get(), new OnlineKeyValidator(), new OfflineKeyValidator(),
                new VaadinComIntegration());
    }

    // For testing
    static void checkLicenseAsync(Product product, BuildType buildType, Callback callback,
            Consumer<String> noKeyUrlHandler,
            String machineId,
            ProKey proKey, OfflineKey offlineKey, OnlineKeyValidator proKeyValidator,
            OfflineKeyValidator offlineKeyValidator, VaadinComIntegration vaadinComIntegration) {
        new Thread(() -> {
            try {
                checkLicense(product, buildType, noKeyUrlHandler, machineId, proKey, offlineKey, proKeyValidator,
                        offlineKeyValidator, vaadinComIntegration);
                callback.ok();
            } catch (Exception e) {
                callback.failed(e);
            }
        }).start();
    }

    /**
     * The internal method all license check methods end up calling.
     * <p>
     * Checking works like:
     * <ol>
     * <li>If there is a local pro key, check with the license server that the
     * proKey
     * has access to the given product
     * <li>If there is no local pro key but there is an offline key, check that this
     * offline key is valid and allows access to the given product
     * <li>If there was neither a local proKey nor an offline key, open the
     * vaadin.com URL for fetching a pro key for the user. Wait for the user to log
     * in, store the pro key and then validate it for the select product like in
     * step 1.
     * </ol>
     * 
     * @param product         the name and version of the product to check
     * @param buildType       the type of build: development or production
     * @param noKeyUrlHandler a handler that is invoked to open the vaadin.com URL
     *                        to download the key file. Used when no key file is
     *                        avialable.
     */
    static void checkLicense(Product product, BuildType buildType, Consumer<String> noKeyUrlHandler) {
        getLogger().debug("Checking license for " + product);
        checkLicense(product, buildType, noKeyUrlHandler, MachineId.get(), LocalProKey.get(), LocalOfflineKey.get(),
                new OnlineKeyValidator(), new OfflineKeyValidator(), new VaadinComIntegration());
    }

    // Version for testing only
    static void checkLicense(Product product, BuildType buildType, Consumer<String> noKeyUrlHandler, String machineId,
            ProKey proKey, OfflineKey offlineKey, OnlineKeyValidator proKeyValidator,
            OfflineKeyValidator offlineKeyValidator, VaadinComIntegration vaadinComIntegration) {
        Result onlineCheckResult = proKeyValidator.validate(product, proKey, machineId, buildType);

        if (onlineCheckResult == Result.OK) {
            // Online validation OK
            // TODO Show name and account
            return;
        } else if (onlineCheckResult == Result.NO_ACCESS && proKey != null) {
            // Online validation was done but the key does not give access to the product
            throw new LicenseException("The provided license key does not allow usage of " + product.getName() + " "
                    + product.getVersion()
                    + ". Check that the license has not expired and that the product you are trying to use does not require another type of subscription.");
        }

        if (offlineKeyValidator.validate(product, buildType, offlineKey, machineId)) {
            // Offline validation OK
            if (loggedLicenseOwner == null && offlineKey.getName() != null) {
                // Show this message only once to avoid spamming once per product

                LocalDateTime expires = LocalDateTime.ofEpochSecond(offlineKey.getExpires()/1000L, 0, ZoneOffset.UTC);
                long expiresInDays = Duration.between(LocalDateTime.now(), expires).toDays();
                loggedLicenseOwner = "Using offline license registered to " + offlineKey.getName() + " / "
                        + offlineKey.getAccount() + ". Expires in " + expiresInDays + " days.";
                getLogger().info(loggedLicenseOwner);
            }
            return;
        }

        if (onlineCheckResult == Result.CANNOT_REACH_SERVER) {
            if (History.isRecentlyValidated(product, Period.ofDays(2), buildType)) {
                // Offline but has validated during the last 2 days. Allow
                return;
            } else if (strictOffline) {
                // Need an offline license
                throw new LicenseException(OfflineKeyValidator.getMissingOfflineKeyMessage(machineId));
            } else {
                // Allow usage offline
                getLogger().warn(
                        "Unable to validate the license, please check your internet connection");
                return;

            }
        }

        if (proKey == null) {
            // No proKey found - probably first launch

            proKey = vaadinComIntegration.openBrowserAndWaitForKey(product, noKeyUrlHandler);
            if (proKey != null) {
                LocalProKey.write(proKey);
                if (proKeyValidator.validate(product, proKey, machineId, buildType) == Result.OK) {
                    // Online validation OK
                    return;
                }
            }
        }

        throw new LicenseException(
                "Unable to validate the license, please check your internet connection. If you need to work offline then "
                        + OfflineKeyValidator.getOfflineKeyLinkMessage(machineId));

    }

    public static Logger getLogger() {
        return LoggerFactory.getLogger(LicenseChecker.class);
    }

    public static void setStrictOffline(boolean strictOffline) {
        LicenseChecker.strictOffline = strictOffline;
    }

}
