/**
 * Copyright (C) 2022 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.ECPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.text.ParseException;
import java.time.Instant;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.ECDSAVerifier;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.jwt.SignedJWT;

/**
 * Validator that can allow or deny usage of a given product version for a given
 * offline key.
 * <p>
 * For development, the offline key can be used if it has not expired and the
 * checksum matches the expected for the machine it is run on.
 * <p>
 * For production, the offline key can also be used if it is a productionOnly
 * key and has not expired.
 */
public class OfflineKeyValidator {

    private static final Map<String, String> PUBLIC_KEYS = new HashMap<>();
    static {
        // Staging
        PUBLIC_KEYS.put("b98c7421853a2d11fb2be2fb73f89be54414a4e9",
                "MIGbMBAGByqGSM49AgEGBSuBBAAjA4GGAAQB/qMpeWPlOKTd8+93GSpi3s/CQMx1" +
                        "gpkw728vl8iijo2965zIBD1bePNULT9VK1iul2iNJA2ev9ImXecLAA4UoMwAlz3t" +
                        "QHIA8zJksNbQUHZhzS74hH/jJr9pE6ra4Q3lnNvmJKEXkFvCpUoBmdYS94Hu0MFX" +
                        "Fi16IJfooLW6qzmtUGs=");
        // Production
        PUBLIC_KEYS.put("542764e7000908e65dc3fc1dabf4e2cd28966758",
                "MIGbMBAGByqGSM49AgEGBSuBBAAjA4GGAAQABaxDhdMljdpoM43y31co033oQjTZ" +
                        "oCj+Wjby9LRBPmdlvMTAJV6gXOzZHDpXQb4N1O0NJr4AeXxaE4GO/p4GGywAkg+S" +
                        "YIO1v8X+n2beq1czN+i8WL1cfUu8DFITUkSHtULtPyNTvW1Ew7XeTGVUQ6n/Xz2Y" +
                        "fAy7tcoFDsldrurE1nY=");
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    private static boolean isExpired(long expires) {
        return Instant.now().isAfter(Instant.ofEpochMilli(expires));
    }

    /**
     * Validates that the given offline license is valid and provides access to the
     * given product.
     * 
     * @param product    the product to validate
     * @param buildType  the type of build: production or development
     * @param offlineKey the offline license key
     * @param machineId  the id of the machine where we are running
     * 
     * @return {@code true} if the validation succeeded
     * @throws LicenseException if the validation fails because and invalid offline
     *                          key was provided
     */
    boolean validate(Product product, BuildType buildType, OfflineKey offlineKey, String machineId) {
        getLogger().debug("Offline validation using offlineKey for " + product);
        if (offlineKey == null) {
            getLogger().debug("No offline key found");
            return false;
        }
        if (History.isRecentlyValidated(product, buildType)) {
            // check only every 24h
            getLogger().debug(
                    "Skipping check as product license was recently validated.");
            return true;
        }

        validateOfflineKey(offlineKey, machineId);

        if (buildType != null) {
            // When buildType is null, usage should be allowed regardless of
            // production/development limitations in the license
            if (buildType == BuildType.DEVELOPMENT && !offlineKey.isDevelopmentBuildAllowed()) {
                getLogger().debug("Offline key is not for development");
                throw new LicenseException(getNotDevelopmentMessage(machineId));

            }
            if (buildType == BuildType.PRODUCTION && !offlineKey.isProductionBuildAllowed()) {
                getLogger().debug("Offline key is not for production builds");
                throw new LicenseException(getNotProductionBuildMessage(machineId));
            }
        }

        if (offlineKey.getMachineId() != null) {
            // If there is an machine id, it must match. Otherwise it is a special, broad
            // offline license
            if (!machineId.equals(offlineKey.getMachineId())) {
                getLogger().debug("Offline key has incorrect machine id");
                throw new LicenseException(getInvalidOfflineKeyMessage(machineId));
            }
        }

        if (isExpired(offlineKey.getExpires())) {
            getLogger().debug("Offline key expired");
            throw new LicenseException(getExpiredOfflineKeyMessage(machineId));
        }

        if (!containsProduct(offlineKey, product)) {
            throw new LicenseException(
                    "The offline key does not provide access to " + product.getName() + " " + product.getVersion());
        }
        History.setLastCheckTimeNow(product, buildType);
        History.setLastSubscription(product,
                offlineKey.getSubscription());

        getLogger().debug("Offline key OK");
        return true;
    }

    private boolean containsProduct(OfflineKey offlineKey, Product product) {
        if (offlineKey.getAllowedProducts().contains(product.getName())) {
            return true;
        }
        if (product.getName().startsWith("test-")) {
            // Allow all test products to pass
            return true;
        }
        if (product.getName().equals("vaadin-framework")) {
            if (product.getVersion().startsWith("7") && offlineKey.getAllowedFeatures().contains("v7extendedsupport")) {
                return true;
            }
            if (product.getVersion().startsWith("8") && offlineKey.getAllowedFeatures().contains("v8extendedsupport")) {
                return true;
            }

        }
        return false;

    }

    private void validateOfflineKey(OfflineKey offlineKey, String machineId) {
        try {
            String jwtData = offlineKey.getJwtData();

            // Claim kid = JWT.decode(jwtData).getHeaderClaim("kid");
            SignedJWT jwt = (SignedJWT) JWTParser.parse(jwtData);
            Object kid = jwt.getHeader().getKeyID();
            KeyFactory fact = KeyFactory.getInstance("EC");
            byte[] encoded = Base64.getDecoder().decode(PUBLIC_KEYS.get(kid));
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(encoded);
            ECPublicKey publicKey = (ECPublicKey) fact.generatePublic(keySpec);

            JWSVerifier verifier = new ECDSAVerifier(publicKey);
            if (!jwt.verify(verifier)) {
                getLogger().debug("Offline key failed verification");
                throw new LicenseException(getInvalidOfflineKeyMessage(machineId));
            }
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            getLogger().debug("Offline key could not be read", e);
            throw new LicenseException(getErrorValidatingOfflineKeyMessage(machineId), e);
        } catch (ParseException e) {
            getLogger().debug("Error parsing offline key", e);
            throw new LicenseException(getInvalidOfflineKeyMessage(machineId), e);
        } catch (JOSEException e) {
            getLogger().debug("Error reading offline key", e);
            throw new LicenseException(getInvalidOfflineKeyMessage(machineId), e);
        }

    }

    private static String getExpiredOfflineKeyMessage(String machineId) {
        return "Offline key has expired, " + getOfflineKeyLinkMessage(machineId);
    }

    private static String getNotDevelopmentMessage(String machineId) {
        return "The provided offline key does not allow development, " + getOfflineKeyLinkMessage(machineId);
    }

    private static String getNotProductionBuildMessage(String machineId) {
        return "The provided offline key does not allow production builds, " + getOfflineKeyLinkMessage(machineId);
    }

    static String getOfflineKeyLinkMessage(String machineId) {
        return "please go to " + getOfflineUrl(machineId) + " to retrieve an offline key."
                + " For troubleshooting steps, see https://vaadin.com/licensing-faq-and-troubleshooting.";
    }

    public static String getOfflineUrl(String machineId) {
        return "https://vaadin.com/pro/validate-license?getOfflineKey="
                + machineId;
    }

    private static String getErrorValidatingOfflineKeyMessage(String machineId) {
        return "Unable to validate offline key, " + getOfflineKeyLinkMessage(machineId);
    }

    static String getInvalidOfflineKeyMessage(String machineId) {
        return "Invalid offline key, " + getOfflineKeyLinkMessage(machineId);
    }

    static String getMissingOfflineKeyMessage(String machineId) {
        return "No offline key, " + getOfflineKeyLinkMessage(machineId);
    }

}
