package com.vaadin.pro.licensechecker;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;

import org.slf4j.Logger;

public class LocalProKey {

    private static ProKey read(File proKeyLocation) throws IOException {
        if (!proKeyLocation.exists()) {
            return null;
        }
        try (FileInputStream is = new FileInputStream(proKeyLocation)) {
            return ProKey.fromJson(Util.toString(is));
        }
    }

    public static File getVaadinFolder() {
        String userHome = System.getProperty("user.home");
        return new File(userHome, ".vaadin");
    }

    public static File getLocation() {
        return new File(getVaadinFolder(), "proKey");
    }

    public static ProKey get() {
        ProKey proKey = getSystemProperty();
        if (proKey != null) {
            getLogger().debug("Using proKey from system property");
            return proKey;
        }
        proKey = getEnvironmentVariable();
        if (proKey != null) {
            getLogger().debug("Using proKey from environment variable");
            return proKey;
        }
        File proKeyLocation = getLocation();
        try {
            proKey = read(proKeyLocation);
            if (proKey != null) {
                getLogger().debug("Found proKey in " + proKeyLocation);
            }
            return proKey;
        } catch (IOException e) {
            getLogger().debug("Unable to read proKey", e);
            return null;
        }
    }

    private static ProKey getSystemProperty() {
        String value = System.getProperty("vaadin.proKey");
        if (value == null) {
            value = System.getProperty("vaadin.key");
            if (!isProKey(value)) {
                return null;
            }
        }
        return parseProKey(value, "system property");
    }

    private static ProKey getEnvironmentVariable() {
        String value = EnvironmentVariables.get("VAADIN_PRO_KEY");
        if (value == null) {
            value = EnvironmentVariables.get("VAADIN_KEY");
            if (!isProKey(value)) {
                return null;
            }
        }

        return parseProKey(value, "environment variable");
    }

    private static ProKey parseProKey(String value, String source) {
        String username = "";
        String key = "";

        if (value.contains("/")) {
            String[] parts = value.split("/");
            username = parts[0];
            key = parts[1];
        } else if (value.startsWith("pro-")) {
            key = value;
        }
        if (key == null || key.isEmpty()) {
            getLogger().warn("Invalid pro key format in the " + source
                    + ". The value must be either of type [vaadin.com login email]/[pro-123] or [pro-123]. Was "
                    + value);
            return null;
        }

        return new ProKey(username, key);
    }

    private static boolean isProKey(String value) {
        if (value == null) {
            return false;
        }
        return value.contains("pro-");
    }

    public static void write(ProKey proKey) {
        File proKeyLocation = getLocation();
        File proKeyDirectory = getLocation().getParentFile();
        if (!proKeyDirectory.exists()) {
            proKeyDirectory.mkdirs();
        }
        getLogger().debug("Writing pro key to " + proKeyLocation);
        try (FileWriter writer = new FileWriter(proKeyLocation)) {
            writer.write(proKey.toJson());
        } catch (IOException e) {
            getLogger().error("Unable to write key to " + proKeyLocation, e);
        }

    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

}
