package com.vaadin.pro.licensechecker;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.time.Instant;
import java.util.UUID;
import java.util.function.Consumer;

import org.slf4j.Logger;

public class VaadinComIntegration {

    private static final String BASE_URL = "https://vaadin.com/pro/validate-license";
    private static final String MANUAL_PRO_KEY_URL = BASE_URL;

    ProKey openBrowserAndWaitForKey(Product product,
            Consumer<String> noKeyUrlHandler, int timeoutSeconds) {
        String uniqueId = UUID.randomUUID().toString();
        String loginUrl = constructUrl(uniqueId, product);

        noKeyUrlHandler.accept(loginUrl);

        try {
            return listenForProKey(new URL(BASE_URL + "/connect/" + uniqueId),
                    timeoutSeconds);
        } catch (MalformedURLException e) {
            throw new LicenseException("Internal error: Invalid URL");
        }
    }

    public static String constructUrl(String uniqueId, Product product) {
        String url = BASE_URL + "?productName=" + product.getName()
                + "&productVersion=" + product.getVersion();
        if (uniqueId != null) {
            url += "&connect=" + uniqueId;
        }
        return url;
    }

    ProKey listenForProKey(URL url, int timeoutSeconds) {
        Instant timeout = Instant.now().plusSeconds(timeoutSeconds);
        while (Instant.now().isBefore(timeout)) {
            getLogger().debug("Polling server for pro key");
            try (InputStream in = url.openStream()) {
                String data = Util.toString(in);
                return ProKey.fromJson(data);
            } catch (FileNotFoundException e) {
                // Server returns 404 if the login process has not completed
            } catch (Exception e) {
                // If the network is bad, a lot of different exceptions can
                // happen
            }
            // Keep trying
            try {
                Thread.sleep(500);
            } catch (InterruptedException e1) {
            }

        }
        return null;
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }
}
