/**
 * Copyright (C) 2022 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import java.text.ParseException;
import java.util.Collections;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTParser;

/**
 * A key for offline validation of licenses.
 * <p>
 * An offline key contains information about the user, the subscription and has
 * an expiration date. It is machine specific and tied to the value returned by
 * {@link MachineId#get()}).
 * <p>
 * A special version of an offline key is only for production builds. This key
 * is not tied to the machine id as it is meant for CI servers where the machine
 * id can change between builds. A normal offline key should always be used for
 * development.
 * <p>
 * This version has no product information in the offline key but only validates
 * that an active subscription exists.
 * <p>
 * An offline key is acquired by providing the machine id to a page on
 * vaadin.com
 */
public class OfflineKey {
    private String jwtData;
    private JWT jwt;

    public OfflineKey(String jwtData) throws ParseException {
        this.jwtData = jwtData;
        this.jwt = JWTParser.parse(jwtData);
    }

    public String getJwtData() {
        return jwtData;
    }

    public String getSubscription() {
        try {
            return jwt.getJWTClaimsSet().getStringClaim("subscription");
        } catch (ParseException e) {
            getLogger().debug(
                    "Unable to parse claim from JWT. Should not happen", e);
            return null;
        }
    }

    public String getName() {
        try {
            return jwt.getJWTClaimsSet().getStringClaim("name");
        } catch (ParseException e) {
            getLogger().debug(
                    "Unable to parse claim from JWT. Should not happen", e);
            return null;
        }
    }

    public String getAccount() {
        try {
            return jwt.getJWTClaimsSet().getStringClaim("account");
        } catch (ParseException e) {
            getLogger().debug(
                    "Unable to parse claim from JWT. Should not happen", e);
            return null;
        }
    }

    public long getExpires() {
        try {
            return jwt.getJWTClaimsSet().getExpirationTime().getTime();
        } catch (ParseException e) {
            getLogger().debug(
                    "Unable to parse claim from JWT. Should not happen", e);
            return 0L;
        }
    }

    public boolean isProductionBuildAllowed() {
        return hasBuildType(BuildType.PRODUCTION.getKey());
    }

    public boolean isDevelopmentBuildAllowed() {
        return hasBuildType(BuildType.DEVELOPMENT.getKey());
    }

    private boolean hasBuildType(String key) {
        try {
            List<String> buildType = jwt.getJWTClaimsSet()
                    .getStringListClaim("build_types");
            if (buildType != null) {
                return buildType.contains(key);
            }
        } catch (ParseException e) {
            getLogger().debug(
                    "Unable to parse claim from JWT. Should not happen", e);
        }
        return true;
    }

    public String getMachineId() {
        try {
            String machineId = jwt.getJWTClaimsSet()
                    .getStringClaim("machine_id");
            // The machineId field is supposed to be missing but some license
            // were
            // incorrectly created with an empty value instead
            if (machineId == null || machineId.isEmpty()) {
                return null;
            } else {
                return machineId;
            }
        } catch (ParseException e) {
            getLogger().debug(
                    "Unable to parse claim from JWT. Should not happen", e);
            return null;
        }
    }

    public List<String> getAllowedProducts() {
        try {
            List<String> list = jwt.getJWTClaimsSet()
                    .getStringListClaim("allowedProducts");
            if (list != null) {
                return list;
            }
        } catch (ParseException e) {
            getLogger().debug(
                    "Unable to parse claim from JWT. Should not happen", e);
        }
        return Collections.emptyList();
    }

    public List<String> getAllowedFeatures() {
        try {
            List<String> list = jwt.getJWTClaimsSet()
                    .getStringListClaim("allowedFeatures");
            if (list != null) {
                return list;
            }
        } catch (ParseException e) {
            getLogger().debug(
                    "Unable to parse claim from JWT. Should not happen", e);
        }
        return Collections.emptyList();
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

}
