/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLConnection;
import java.net.UnknownHostException;

import javax.net.ssl.SSLHandshakeException;

import org.slf4j.Logger;

import elemental.json.Json;
import elemental.json.JsonObject;

public class OnlineKeyValidator {

    private static final String UNABLE_TO_VALIDATE_SUBSCRIPTION = "Unable to validate subscription."
            + " Please go to https://vaadin.com/pro/validate-license to check that your subscription is active."
            + " For troubleshooting steps, see https://vaadin.com/licensing-faq-and-troubleshooting.";
    static final String LICENSE_VALIDATION_HOST = "https://tools.vaadin.com/";
    private static final String LICENSE_VALIDATION_URL = LICENSE_VALIDATION_HOST
            + "vaadin-license-server/licenses/pro";
    private static final String[] PROPERTIES = new String[] { "java.vendor",
            "java.version", "os.arch", "os.name", "os.version" };

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    public static enum Result {
        OK, NO_ACCESS, CANNOT_REACH_SERVER
    }

    Result validate(Product product, ProKey proKey, String machineId,
            BuildType buildType, boolean quiet) {
        getLogger().debug("Online validation using proKey for " + product);
        if (proKey == null) {
            getLogger().debug("No pro key found");
            return Result.NO_ACCESS;
        }
        if (History.isRecentlyValidated(product, buildType, proKey)) {
            // check only every 24h
            getLogger().debug(
                    "Skipping check as product license was recently validated.");
            return Result.OK;
        }
        try {
            JsonObject response = queryServer(product, proKey, machineId,
                    buildType);
            if (validateServerResponse(product, response, quiet)) {
                History.setLastCheckTimeNow(product, buildType, proKey);
                History.setLastSubscription(product,
                        response.getString("subscription"), proKey);
                getLogger().debug("Pro key OK");
                return Result.OK;
            } else {
                // Key does not have access to the given product
                // Subscription might have ended or some other problem
                getLogger().debug("Pro key does not have access");
                return Result.NO_ACCESS;
            }
        } catch (UnknownHostException | ConnectException
                | SocketTimeoutException e) {
            getLogger().debug("Pro key unable to connect to server");
            return Result.CANNOT_REACH_SERVER;
        } catch (SSLHandshakeException e) {
            getLogger().warn("Pro key unable to connect to server", e);
            return Result.CANNOT_REACH_SERVER;
        } catch (Exception e) {
            if (!quiet) {
                getLogger().error(UNABLE_TO_VALIDATE_SUBSCRIPTION, e);
            }
            getLogger().debug("Pro key checking failed with exception", e);
            return Result.NO_ACCESS;
        }
    }

    /**
     * @return subscription (Trial, Pro, ...) if success, null otherwise
     */
    static String getSubscription(Product product, ProKey proKey) {
        if (proKey == null) {
            getLogger().debug("No pro key found to get subscription");
            return null;
        }
        return History.getLastSubscription(product, proKey);
    }

    private static boolean validateServerResponse(Product product,
            JsonObject response, boolean quiet) {
        getLogger().debug("Validating license for " + product.getName() + " "
                + product.getVersion());
        String result = response.getString("result");
        String message = response.getString("message");
        if (!quiet && message != null && !message.isEmpty()) {
            getLogger().info(message);
        }
        return "ok".equals(result);
    }

    private static JsonObject queryServer(Product product, ProKey proKey,
            String machineId, BuildType buildType) throws IOException {
        URL url = new URL(LICENSE_VALIDATION_URL);
        URLConnection http = url.openConnection();
        http.setRequestProperty("check-source", "java");
        http.setRequestProperty("machine-id", machineId);
        Platform.getVaadinVersion().ifPresent(version -> {
            http.setRequestProperty("vaadin-version", version);
        });
        http.setRequestProperty("product-name", product.getName());
        http.setRequestProperty("product-version", product.getVersion());
        if (buildType != null) {
            http.setRequestProperty("build-type", buildType.getKey());
        }
        for (String property : PROPERTIES) {
            String value = System.getProperty(property);
            if (value != null) {
                http.setRequestProperty("prop-" + property.replace(".", "-"),
                        value);
            }
        }

        http.setRequestProperty("product-version", product.getVersion());
        http.setRequestProperty("Cookie", "proKey=" + proKey.getProKey());
        http.setConnectTimeout(5000);
        http.setReadTimeout(5000);
        http.connect();

        try (InputStream in = http.getInputStream()) {
            String response = Util.toString(in);
            return Json.parse(response);
        }
    }

}
