/**
 * Copyright (C) 2022 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.pro.licensechecker;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.text.ParseException;
import java.util.Optional;
import java.util.stream.Stream;

import org.slf4j.Logger;

/**
 * Fetches an offline key from either the home directory (~/.vaadin/offlineKey)
 * or from a system property ({@code vaadin.offlineKey}) if available. Also
 * capable of writing an offline key but this should not be used except for
 * testing. Offline keys are always downloaded and stored manually.
 */
public class LocalOfflineKey {

    private static OfflineKey read(File offlineKeyLocation) throws IOException {
        if (!offlineKeyLocation.exists()) {
            return null;
        }
        try (FileInputStream is = new FileInputStream(offlineKeyLocation)) {
            OfflineKey key = new OfflineKey(Util.toString(is));
            return key;
        } catch (ParseException e) {
            getLogger().error("Unable to read offline license from file", e);
            return null;
        }
    }

    public static File[] getLocations() {
        return Stream
                .of("offlineKey", "offlineKey.txt", "serverKey",
                        "serverKey.txt")
                .map(name -> new File(LocalProKey.getVaadinFolder(), name))
                .toArray(File[]::new);
    }

    /**
     * Finds an offline key, either from a system property or from the home
     * directory.
     *
     * @return an offline key or {@code null} if no key was found
     */
    public static OfflineKey get() {
        OfflineKey offlineKey = getSystemProperty();
        if (offlineKey != null) {
            getLogger().debug("Using offlineKey from system property");
            return offlineKey;
        }
        offlineKey = getEnvironmentVariable();
        if (offlineKey != null) {
            getLogger().debug("Using offlineKey from environment variable");
            return offlineKey;

        }

        try {
            File keyLocation = null;
            for (File location : getLocations()) {
                offlineKey = read(location);
                if (offlineKey != null) {
                    keyLocation = location;
                    break;
                }
            }

            if (offlineKey != null) {
                getLogger().debug("Found offline key in " + keyLocation);
            }
            return offlineKey;
        } catch (IOException e) {
            getLogger().debug("Unable to read offline key", e);
            return null;
        }
    }

    private static OfflineKey getSystemProperty() {
        String jwtData = System.getProperty("vaadin.offlineKey");
        if (jwtData == null) {
            jwtData = System.getProperty("vaadin.key");
            if (jwtData == null) {
                return null;
            }
        }

        try {
            return new OfflineKey(jwtData);
        } catch (ParseException e) {
            getLogger().error(
                    "Unable to read offline license from system property", e);
            return null;
        }
    }

    private static OfflineKey getEnvironmentVariable() {
        String value = EnvironmentVariables.get("VAADIN_OFFLINE_KEY");
        if (value == null) {
            value = EnvironmentVariables.get("VAADIN_KEY");
            if (value == null) {
                return null;
            }
        }
        try {
            return new OfflineKey(value);
        } catch (ParseException e) {
            getLogger().error(
                    "Unable to read offline license from environment variable",
                    e);
            return null;
        }
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

}
