/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker.dau;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

/**
 * Represents a Daily Active User.
 * <p>
 * </p>
 * A Daily Active User is uniquely identified by the tracking hash and the
 * timestamp of the first interaction with the application.
 * <p>
 * </p>
 * To better identify single users, a user identity hash can be set. Instances
 * with same tracking hash and first interaction time but different identity
 * hash are considered different users.
 * <p>
 * </p>
 * Instances should not contain any sensible information. Data have to be given
 * as not revertible salted hashes.
 */
final class TrackedUser {
    private final String trackingHash;
    private final Instant creationTime;
    private final Set<String> userIdentityHashes = new HashSet<>();

    /**
     * Creates a new {@link TrackedUser} for the given a tracking hash.
     *
     * @param trackingHash
     *            the hash of the unique tracking identifier.
     */
    TrackedUser(String trackingHash) {
        this(trackingHash, Instant.now());
    }

    /**
     * Creates a new {@link TrackedUser} for the given tracking hash and
     * creation time.
     * <p>
     * Note: this constructor is mainly used for testing, one should usually use
     * {@link TrackedUser#TrackedUser(String)}
     *
     * @param trackingHash
     *            the hash of the unique tracking identifier.
     * @param creationTime
     *            the zoned time when the user has started working with app.
     */
    TrackedUser(String trackingHash, Instant creationTime) {
        this.trackingHash = Objects.requireNonNull(trackingHash,
                "trackingHash must not be null");
        this.creationTime = (creationTime != null
                ? creationTime.truncatedTo(ChronoUnit.SECONDS)
                : Instant.now().truncatedTo(ChronoUnit.SECONDS));
    }

    /**
     * A hashed identifier that uniquely identifies an application user.
     *
     * @return hashed identifier that uniquely identifies an application user
     */
    public String getTrackingHash() {
        return trackingHash;
    }

    /**
     * Gets the instant of the tracking entry creation.
     *
     * @return the instant of the tracking entry creation.
     */
    public Instant getCreationTime() {
        return creationTime;
    }

    /**
     * Gets the linked user identity hashes.
     *
     * @return the identity hashes associated to this tracked user; can be
     *         empty, but never {@literal null}. The returned set is
     *         unmodifiable.
     */
    public Set<String> getUserIdentityHashes() {
        return Collections.unmodifiableSet(userIdentityHashes);
    }

    /**
     * Associates the given identity with this tracked user.
     * <p>
     * </p>
     * A tracked user can have multiple identities, for example if during the
     * same browser session a user authenticates with different credentials.
     * <p>
     * </p>
     * Null or blank values are ignored.
     *
     * @param userIdentityHash
     *            hashed user identity.
     * @return {@literal true} if the given identity was not already known,
     *         {@literal false} otherwise.
     */
    boolean linkUserIdentity(String userIdentityHash) {
        if (userIdentityHash != null && !userIdentityHash.trim().isEmpty()) {
            return userIdentityHashes.add(userIdentityHash);
        }
        return false;
    }

    /**
     * Gets if the tracked user has linked identities.
     *
     * @return {@literal true} if the tracked user has linked identities,
     *         otherwise {@literal false}.
     */
    boolean hasLinkedIdentities() {
        return !userIdentityHashes.isEmpty();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;

        TrackedUser that = (TrackedUser) o;
        return trackingHash.equals(that.trackingHash)
                && creationTime.equals(that.creationTime)
                && userIdentityHashes.equals(that.userIdentityHashes);
    }

    @Override
    public int hashCode() {
        int result = trackingHash.hashCode();
        result = 31 * result + creationTime.hashCode();
        result = 31 * result + userIdentityHashes.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "TrackedUser{" + "trackingHash='" + trackingHash + '\''
                + ", creationTime=" + creationTime
                + ((!userIdentityHashes.isEmpty())
                        ? ", userIdentityHashes=["
                                + String.join(",", userIdentityHashes) + ']'
                        : "")
                + '}';
    }
}
