/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;
import java.time.Period;
import java.time.temporal.TemporalAmount;
import java.util.prefs.Preferences;

import org.slf4j.Logger;

public class History {

    private static final String CHECK_PREFIX = "lastcheck-";

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    static boolean isRecentlyValidated(Product product, BuildType buildType,
            ProKey proKey) {
        return isRecentlyValidated(product, Period.ofDays(1), buildType,
                proKey);
    }

    static boolean isRecentlyValidated(Product product, TemporalAmount period,
            BuildType buildType, ProKey proKey) {
        getLogger().debug("Checking if license for " + product
                + " has recently been checked for build type " + buildType
                + " and pro key " + proKey);
        Instant lastCheck = getLastCheckTime(product, buildType, proKey);
        if (lastCheck == null) {
            return false;
        }
        Instant now = Instant.now();
        if (lastCheck.isAfter(now)) {
            // Invalid last check value
            return false;
        }

        Instant nextCheck = lastCheck.plus(period);
        return now.isBefore(nextCheck);
    }

    public static Instant getLastCheckTime(Product product, BuildType buildType,
            ProKey proKey) {
        String lastCheckKey = getLastCheckKey(product, buildType, proKey);
        long lastCheck = getPreferences().getLong(lastCheckKey, -1);
        if (lastCheck == -1) {
            getLogger().debug(
                    "License for " + product + " has never been checked");
            return null;
        } else {
            Instant lastCheckInstant = Instant.ofEpochMilli(lastCheck);
            getLogger().debug("Last check for " + product + " was on "
                    + lastCheckInstant);
            return lastCheckInstant;
        }
    }

    public static String getLastSubscription(Product product, ProKey proKey) {
        String lastSubscriptionKey = getLastSubscriptionKey(product, proKey);
        return getPreferences().get(lastSubscriptionKey, "");
    }

    static long setLastCheckTimeNow(Product product, BuildType buildType,
            ProKey proKey) {
        getLogger().debug("Marking license for " + product
                + " as checked now for buildType " + buildType);
        return setLastCheck(product, Instant.now(), buildType, proKey);
    }

    static long setLastCheck(Product product, Instant instant,
            BuildType buildType, ProKey proKey) {
        getLogger().debug("Marking license for " + product + " as checked on "
                + instant + " for buildType " + buildType + " and pro key "
                + proKey);

        long timeMillis = instant.toEpochMilli();
        getPreferences().putLong(getLastCheckKey(product, buildType, proKey),
                timeMillis);
        return timeMillis;
    }

    public static String setLastSubscription(Product product,
            String subscription, ProKey proKey) {
        getLogger().debug("Storing subscription for " + product);

        getPreferences().put(getLastSubscriptionKey(product, proKey),
                subscription);
        return subscription;
    }

    static String getLastCheckKey(Product product, BuildType buildType,
            ProKey proKey) {
        String key = getKey(product, proKey);
        if (buildType != null) {
            key += "-" + buildType.getKey();
        }
        return hash(key);
    }

    static String hash(String key) {
        // We use a hash as there is a max length of 80 characters for the key
        // and we do not really know how long product names and versions will be
        // used
        try {

            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(key.getBytes());
            byte[] digest = md.digest();
            BigInteger no = new BigInteger(1, digest);
            String hash = no.toString(16);
            while (hash.length() < 32) {
                hash = "0" + hash;
            }

            return CHECK_PREFIX + hash;
        } catch (NoSuchAlgorithmException e) {
            // This does not really happen
            return (CHECK_PREFIX + key).substring(0,
                    Preferences.MAX_KEY_LENGTH);
        }

    }

    private static String getKey(Product product, ProKey proKey) {
        String suffix = proKey == null ? "null" : proKey.getProKey();
        return (product != null ? product.getName() + "-" + product.getVersion()
                + "-" : "null-null-") + suffix;
    }

    static String getLastSubscriptionKey(Product product, ProKey proKey) {
        String key = getKey(product, proKey) + "-sub";
        return hash(key);
    }

    private static Preferences getPreferences() {
        return Preferences.userNodeForPackage(OnlineKeyValidator.class);
    }

    /**
     * Clears the local check history so that all products are checked again.
     */
    public static void clearAll() throws Exception {
        Preferences preferences = getPreferences();
        for (String key : preferences.keys()) {
            if (key.startsWith(CHECK_PREFIX)) {
                preferences.remove(key);
            }
        }
    }

}
