package com.vaadin.pro.licensechecker;

import java.time.Instant;
import java.time.Period;
import java.util.logging.Logger;
import java.util.prefs.Preferences;

public class History {

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

    public static boolean isRecentlyValidated(Product product) {
        getLogger().fine("Checking if license for " + product +
                " has recently been checked");
        Instant lastCheck = getLastCheckTime(product);
        if (lastCheck == null) {
            return false;
        }
        Instant now = Instant.now();
        if (lastCheck.isAfter(now)) {
            // Invalid last check value
            return false;
        }

        Instant nextCheck = lastCheck.plus(Period.ofDays(1));
        return now.isBefore(nextCheck);
    }

    public static Instant getLastCheckTime(Product product) {
        String lastCheckKey = getLastCheckKey(product);
        long lastCheck = getPreferences().getLong(lastCheckKey, -1);
        if (lastCheck == -1) {
            getLogger()
                    .fine("License for " + product + " has never been checked");
            return null;
        } else {
            Instant lastCheckInstant = Instant.ofEpochMilli(lastCheck);
            getLogger().fine("Last check for " + product + " was on " +
                    lastCheckInstant);
            return lastCheckInstant;
        }
    }

    public static String getLastSubscription(Product product) {
        String lastSubscriptionKey = getLastSubscriptionKey(product);
        return getPreferences().get(lastSubscriptionKey, "");
    }

    public static long setLastCheckTimeNow(Product product) {
        getLogger().fine("Marking license for " + product + " as checked now");

        long current = Instant.now().toEpochMilli();
        getPreferences().putLong(getLastCheckKey(product), current);
        return current;
    }

    public static String setLastSubscription(Product product,
            String subscription) {
        getLogger().fine("Storing subscription for " + product);

        getPreferences().put(getLastSubscriptionKey(product), subscription);
        return subscription;
    }

    private static String getLastCheckKey(Product product) {
        return "lastcheck-" + product.getName() + "-" + product.getVersion();
    }

    private static String getLastSubscriptionKey(Product product) {
        return getLastCheckKey(product) + "-subscription";
    }

    private static Preferences getPreferences() {
        return Preferences.userNodeForPackage(ProKeyValidator.class);
    }

}
